"""
Configuration management for EvenAge.

Provides pluggable backend configuration via environment and YAML.
All integrations (queue, database, vector, LLM, cache) are configurable.
"""

from __future__ import annotations

from pathlib import Path
from typing import Any, Protocol

from pydantic import BaseModel, Field
from pydantic_settings import BaseSettings
import yaml


class EvenAgeConfig(BaseSettings):
    """
    Environment-based configuration for EvenAge runtime.
    
    Supports pluggable backends for all infrastructure:
    - queue_backend: "redis" (default) | "memory"
    - db_backend: "postgres" (default) | "sqlite"
    - vector_backend: "local" (default) | "pinecone" | "milvus"
    - llm_backend: "noop" (default) | "openai" | "anthropic" | "gemini"
    - cache_backend: "inmemory" (default) | "redis"
    """

    # Pluggable backend selection
    queue_backend: str = Field(default="redis", description="Message queue backend")
    db_backend: str = Field(default="postgres", description="Database backend")
    vector_backend: str = Field(default="local", description="Vector store backend")
    llm_backend: str = Field(default="noop", description="LLM backend (noop=echo)")
    llm_model: str | None = Field(default=None, description="LLM model name (optional, backend-specific)")
    openai_base_url: str | None = Field(default=None, description="OpenAI-compatible base URL (e.g., vLLM)")
    cache_backend: str = Field(default="inmemory", description="Cache backend")

    # Database connection
    database_url: str = Field(
        default="postgresql://postgres:postgres@localhost:5432/evenage",
        description="Database connection URL (postgres or sqlite)",
    )

    # Redis connection (for queue and optional cache)
    redis_url: str = Field(
        default="redis://localhost:6379", description="Redis connection URL"
    )

    # Kafka (optional)
    kafka_bootstrap: str | None = Field(default=None, description="Kafka bootstrap servers (host:port)")

    # MinIO/S3 Storage (optional, for large artifacts)
    minio_endpoint: str = Field(default="localhost:9000", description="MinIO endpoint")
    minio_access_key: str = Field(default="minioadmin", description="MinIO access key")
    minio_secret_key: str = Field(
        default="minioadmin123", description="MinIO secret key"
    )
    minio_secure: bool = Field(default=False, description="Use HTTPS for MinIO")
    minio_bucket: str = Field(default="evenage", description="Default bucket name")

    # Tracing and metrics (stored in DB, not external services)
    enable_tracing: bool = Field(default=True, description="Enable internal tracing to DB")
    enable_metrics: bool = Field(default=True, description="Enable internal metrics collection")

    # Large Response Storage
    enable_large_response_storage: bool = Field(
        default=True, description="Enable automatic storage of large responses in MinIO"
    )
    storage_threshold_kb: int = Field(
        default=100, description="Size threshold in KB for storing responses in MinIO"
    )

    # Inline workers: when running locally via AgentRunner, also spin up
    # lightweight in-process workers to consume delegated tasks without Docker
    enable_inline_workers: bool = Field(
        default=True,
        description="Run inline local workers during AgentRunner.run() so delegated tasks execute without external workers",
    )

    # API
    api_host: str = Field(default="0.0.0.0", description="API server host")
    api_port: int = Field(default=8000, description="API server port")
    api_cors_origins: list[str] = Field(
        default_factory=lambda: ["http://localhost:5173"],
        description="Allowed CORS origins (avoid * in production)"
    )

    # Dashboard URL (for containerized setups)
    dashboard_url: str = Field(
        default="http://localhost:5173",
        description="Dashboard URL for links from API"
    )

    # Agent worker settings
    agent_name: str | None = Field(
        default=None, description="Agent name for worker mode"
    )
    worker_concurrency: int = Field(
        default=1, description="Number of concurrent tasks per worker"
    )

    # Coordinator-Specialist Architecture
    has_coordinator: bool = Field(
        default=False,
        description="Enable coordinator-specialist mode where all results route to coordinator"
    )
    coordinator_agent: str = Field(
        default="coordinator",
        description="Name of the coordinator agent (used when has_coordinator=True)"
    )
    
    # MCP (Model Context Protocol) Integration
    mcp_servers: list[dict[str, str]] = Field(
        default_factory=list,
        description="List of MCP server configs with 'url' and optional 'api_key'"
    )

    # LLM API keys (optional, used when respective llm_backend is selected)
    gemini_api_key: str | None = Field(
        default=None, description="Google Gemini API key"
    )
    openai_api_key: str | None = Field(
        default=None, description="OpenAI API key"
    )
    anthropic_api_key: str | None = Field(
        default=None, description="Anthropic API key"
    )
    groq_api_key: str | None = Field(
        default=None, description="Groq API key"
    )
    http_llm_endpoint: str | None = Field(default=None, description="Generic HTTP LLM endpoint URL")

    # Tool API keys (optional)
    serper_api_key: str | None = Field(
        default=None, description="Serper.dev API key for web search"
    )

    # Dashboard auth (optional)
    evenage_dash_users: str | None = Field(
        default=None, description="Dashboard user credentials (user:pass)"
    )

    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        extra = "ignore"  # Ignore unknown fields from environment


# Backend protocols for pluggable implementations


class QueueBackend(Protocol):
    """Protocol for message queue backends."""

    async def register_agent(self, name: str, metadata: dict) -> bool: ...
    async def publish_task(self, agent_name: str, task: dict) -> str: ...
    async def publish_response(self, task_id: str, response: dict) -> bool: ...
    async def consume_tasks(self, agent_name: str, block_ms: int, count: int) -> list[dict]: ...
    async def wait_for_response(self, task_id: str, timeout_sec: int) -> dict | None: ...
    async def get_registered_agents(self) -> dict[str, dict]: ...
    async def get_queue_depth(self, agent_name: str) -> int: ...
    async def health_check(self) -> bool: ...


class DatabaseBackend(Protocol):
    """Protocol for database backends."""

    def create_job(self, job_id: str, pipeline: str, inputs: dict) -> None: ...
    def save_result(self, job_id: str, outputs: dict) -> None: ...
    def get_result(self, job_id: str) -> dict | None: ...
    def list_jobs(self, limit: int) -> list[dict]: ...
    def append_trace(self, job_id: str, agent_name: str, event_type: str, payload: dict) -> None: ...
    def memory_put(self, agent_name: str, key: str, value: Any, job_id: str | None = None) -> None: ...
    def memory_get(self, agent_name: str, key: str, job_id: str | None = None) -> Any: ...
    def register_agent(self, agent_name: str, metadata: dict) -> None: ...
    def list_agents(self) -> list[dict]: ...


class VectorBackend(Protocol):
    """Protocol for vector store backends."""

    def store(self, key: str, vector: list[float], metadata: dict) -> None: ...
    def search(self, vector: list[float], k: int) -> list[dict]: ...
    def delete(self, key: str) -> None: ...


class LLMBackend(Protocol):
    """Protocol for LLM backends."""

    async def generate(self, prompt: str, system: str | None = None, **kwargs) -> str: ...


class CacheBackend(Protocol):
    """Protocol for cache backends."""

    def get(self, key: str) -> Any: ...
    def set(self, key: str, value: Any, ttl: int | None = None) -> None: ...
    def delete(self, key: str) -> None: ...
    def clear(self) -> None: ...


class BackendFactory:
    """
    Factory for creating backend instances based on configuration.
    
    Provides pluggable implementations for queue, database, vector, LLM, and cache.
    """

    def __init__(self, config: EvenAgeConfig):
        self.config = config

    # Singleton memory bus shared across agents within a process
    _memory_bus_singleton = None

    def create_queue_backend(self) -> QueueBackend:
        """Create message queue backend based on config."""
        if self.config.queue_backend == "redis":
            from .message_bus import RedisBus
            bus = RedisBus(self.config.redis_url)
        elif self.config.queue_backend == "memory":
            from .message_bus import MemoryBus
            if BackendFactory._memory_bus_singleton is None:
                BackendFactory._memory_bus_singleton = MemoryBus()
            bus = BackendFactory._memory_bus_singleton
        elif self.config.queue_backend == "kafka":
            from .message_bus import KafkaBus
            if not self.config.kafka_bootstrap:
                raise ValueError("kafka_bootstrap is required for kafka queue_backend")
            bus = KafkaBus(self.config.kafka_bootstrap)
        else:
            raise ValueError(f"Unknown queue_backend: {self.config.queue_backend}")

        # Optionally enable large-response offload to storage
        try:
            if getattr(self.config, "enable_large_response_storage", False):
                storage = self.create_storage_backend()
                bus._storage = storage
                bus._storage_bucket = self.config.minio_bucket
                bus._storage_threshold_kb = self.config.storage_threshold_kb
        except Exception:
            # Non-fatal: continue without offload
            pass

        return bus

    def create_database_backend(self) -> DatabaseBackend:
        """Create database backend based on config."""
        from .database import DatabaseService
        return DatabaseService(self.config.database_url)

    def create_vector_backend(self) -> VectorBackend:
        """Create vector store backend based on config."""
        if self.config.vector_backend == "local":
            from .vector import LocalVectorStore
            return LocalVectorStore()
        if self.config.vector_backend == "pgvector":
            from .vector import PostgresVectorStore
            return PostgresVectorStore(self.config.database_url)
        raise ValueError(f"Unknown vector_backend: {self.config.vector_backend}")

    def create_llm_backend(self) -> LLMBackend:
        """Create LLM backend based on config."""
        if self.config.llm_backend == "noop":
            from .llm import NoOpLLM
            return NoOpLLM()
        if self.config.llm_backend == "openai":
            from .llm import OpenAILLM
            return OpenAILLM(self.config.openai_api_key, self.config.llm_model, base_url=self.config.openai_base_url)
        if self.config.llm_backend == "anthropic":
            from .llm import AnthropicLLM
            return AnthropicLLM(self.config.anthropic_api_key, self.config.llm_model)
        if self.config.llm_backend == "gemini":
            from .llm import GeminiLLM
            return GeminiLLM(self.config.gemini_api_key, self.config.llm_model)
        if self.config.llm_backend == "http":
            from .llm import HTTPLLM
            if not self.config.http_llm_endpoint:
                raise ValueError("http_llm_endpoint is required for http llm_backend")
            return HTTPLLM(self.config.http_llm_endpoint)
        if self.config.llm_backend == "vllm":
            from .llm import OpenAILLM
            if not self.config.openai_base_url:
                raise ValueError("openai_base_url is required for vllm llm_backend")
            return OpenAILLM(self.config.openai_api_key, self.config.llm_model, base_url=self.config.openai_base_url)
        raise ValueError(f"Unknown llm_backend: {self.config.llm_backend}")

    def create_cache_backend(self) -> CacheBackend:
        """Create cache backend based on config."""
        if self.config.cache_backend == "inmemory":
            from .cache import InMemoryCache
            return InMemoryCache()
        if self.config.cache_backend == "redis":
            from .cache import RedisCache
            return RedisCache(self.config.redis_url)
        raise ValueError(f"Unknown cache_backend: {self.config.cache_backend}")

    def create_storage_backend(self):
        """Create storage backend for artifacts (MinIO/S3)."""
        if self.config.minio_endpoint:
            from .storage import MinIOStorage
            return MinIOStorage(
                endpoint=self.config.minio_endpoint,
                access_key=self.config.minio_access_key,
                secret_key=self.config.minio_secret_key,
                secure=self.config.minio_secure
            )
        # Fall back to in-memory storage if no MinIO configured
        from .storage import InMemoryStorage
        return InMemoryStorage()



class ProjectConfig(BaseModel):
    """
    Project configuration from evenage.yml.
    
    Defines which agents exist and backend preferences.
    """

    name: str = Field(description="Project name")

    # Backend selections (can override env defaults)
    queue_backend: str = Field(default="redis", description="Message queue backend")
    db_backend: str = Field(default="postgres", description="Database backend")
    vector_backend: str = Field(default="local", description="Vector store backend")
    llm_backend: str = Field(default="noop", description="LLM backend")
    cache_backend: str = Field(default="inmemory", description="Cache backend")

    # Agent registry
    agents: list[str] = Field(default_factory=list, description="List of agent names")


class AgentConfig(BaseModel):
    """
    Agent configuration (minimal for @actor pattern).
    
    Most runtime config is auto-discovered or comes from EvenAgeConfig.
    """

    name: str
    role: str
    goal: str
    backstory: str | None = None
    max_iterations: int = Field(default=15)
    allow_delegation: bool = Field(default=True)
    verbose: bool = Field(default=False)


class PipelineStage(BaseModel):
    """Stage definition in a pipeline (simplified)."""

    agent: str = Field(description="Agent name to execute")
    inputs: dict[str, Any] = Field(default_factory=dict, description="Stage inputs")


class PipelineConfig(BaseModel):
    """Pipeline configuration for sequential agent execution."""

    name: str
    description: str | None = None
    stages: list[PipelineStage]


def load_project_config(path: Path | str = "evenage.yml") -> ProjectConfig:
    """Load project configuration from YAML file."""
    path = Path(path)
    if not path.exists():
        raise FileNotFoundError(f"Configuration file not found: {path}")

    with open(path) as f:
        data = yaml.safe_load(f)

    return ProjectConfig(**data.get("project", {}))


def save_project_config(config: ProjectConfig, path: Path | str = "evenage.yml") -> None:
    """Save project configuration to YAML file."""
    path = Path(path)
    data = {"project": config.model_dump(exclude_none=True)}

    with open(path, "w") as f:
        yaml.dump(data, f, default_flow_style=False, sort_keys=False)


def load_pipeline_config(path: Path | str) -> PipelineConfig:
    """Load pipeline configuration from YAML file."""
    path = Path(path)
    if not path.exists():
        raise FileNotFoundError(f"Pipeline configuration not found: {path}")

    with open(path) as f:
        data = yaml.safe_load(f)

    return PipelineConfig(**data)


# --- Code-first infra convenience ---


def connect_infrastructure(config: EvenAgeConfig):
    """
    Explicitly connect all core infrastructure from a config instance.

    Returns a lightweight object with attributes:
    bus, database, vector, llm, storage, cache
    """
    from evenage.infra import Infra
    from evenage.infra.database import connect_database
    from evenage.infra.llm_client import connect_llm
    from evenage.infra.message_bus import connect_bus
    from evenage.infra.storage import connect_storage
    from evenage.infra.vector_store import connect_vector_store

    database = connect_database(config.database_url)

    storage = connect_storage(
        endpoint=config.minio_endpoint,
        access_key=config.minio_access_key,
        secret_key=config.minio_secret_key,
        secure=config.minio_secure,
        fallback_inmemory=True,
    )

    bus = connect_bus(
        config.queue_backend,
        redis_url=config.redis_url if config.queue_backend == "redis" else None,
        kafka_bootstrap=config.kafka_bootstrap if config.queue_backend == "kafka" else None,
        storage=storage if config.enable_large_response_storage else None,
        storage_bucket=config.minio_bucket if config.enable_large_response_storage else None,
        storage_threshold_kb=(config.storage_threshold_kb if config.enable_large_response_storage else None),
    )

    vector = None
    try:
        vector = connect_vector_store(
            config.vector_backend,
            database_url=(config.database_url if config.vector_backend == "pgvector" else None),
        )
    except Exception:
        vector = None

    llm = None
    try:
        llm = connect_llm(
            config.llm_backend,
            model=config.llm_model,
            openai_api_key=config.openai_api_key,
            openai_base_url=config.openai_base_url,
            anthropic_api_key=config.anthropic_api_key,
            gemini_api_key=config.gemini_api_key,
            http_endpoint=config.http_llm_endpoint,
        )
    except Exception:
        llm = None

    cache = BackendFactory(config).create_cache_backend()

    return Infra(
        bus=bus,
        database=database,
        vector=vector,
        llm=llm,
        storage=storage,
        cache=cache,
    )

