from __future__ import annotations

import hashlib
from typing import TYPE_CHECKING

import pytest

if TYPE_CHECKING:
    from types import ModuleType

    from pytest_benchmark.fixture import BenchmarkFixture


@pytest.mark.parametrize(
    ("data", "digest"),
    [
        pytest.param(
            "ff0c4c5a3420636f6d7072657373696f6e207465737420737472696e671b00db507472696e67",
            "5ec59b1b60247178b260e145ae888c318e1fa3ee4466cf0e145de8f24c4b1501",
            id="basic",
        ),
        pytest.param(
            "ffffa94c6f72656d20697073756d20646f6c6f722073697420616d657420636f"
            "6e73656374657475722061646970697363696e6720656c69742e205175697371"
            "75652066617563696275732065782073617069656e2076697461652070656c6c"
            "656e7465737175652073656d20706c6163657261742e20496e20696420637572"
            "737573206d69207072657469756d2074656c6c7573206475697320636f6e7661"
            "6c6c69732e2054656d707573206c656f2065752061656e65616e207365642064"
            "69616d2075726e612074656d706f722e2050756c76696e617220766976616d75"
            "73206672696e67696c6c61206c61637573206e6563206d657475732062696265"
            "6e64756d20656765737461732e20496163756c6973206d61737361206e69736c"
            "206d616c657375616461206c6163696e696120696e7465676572206e756e6320"
            "706f73756572652e2055742068656e6472657269742073656d7065722076656c"
            "20636c61737320617074656e742074616369746920736f63696f7371752e2041"
            "64206c69746f726120746f727175656e742070657220636f6e75626961206e6f"
            "7374726120696e636570746f732068696d656e61656f732e0a0ab701ffffffff"
            "ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff"
            "ffffffffffffffffffffffffffffffffff4550656f732e0a",
            "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529",
            id="large",
        ),
    ],
)
def test_lz4_decompress(lz4: ModuleType, data: str, digest: str) -> None:
    assert hashlib.sha256(lz4.decompress(bytes.fromhex(data))).hexdigest() == digest


@pytest.mark.benchmark
def test_benchmark_lz4_decompress(lz4: ModuleType, benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex("ff0c4c5a3420636f6d7072657373696f6e207465737420737472696e671b00db507472696e67")
    assert benchmark(lz4.decompress, buf) == b"LZ4 compression test string" * 10


@pytest.mark.benchmark
def test_benchmark_large_lz4_decompress(lz4: ModuleType, benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "ffffa94c6f72656d20697073756d20646f6c6f722073697420616d657420636f"
        "6e73656374657475722061646970697363696e6720656c69742e205175697371"
        "75652066617563696275732065782073617069656e2076697461652070656c6c"
        "656e7465737175652073656d20706c6163657261742e20496e20696420637572"
        "737573206d69207072657469756d2074656c6c7573206475697320636f6e7661"
        "6c6c69732e2054656d707573206c656f2065752061656e65616e207365642064"
        "69616d2075726e612074656d706f722e2050756c76696e617220766976616d75"
        "73206672696e67696c6c61206c61637573206e6563206d657475732062696265"
        "6e64756d20656765737461732e20496163756c6973206d61737361206e69736c"
        "206d616c657375616461206c6163696e696120696e7465676572206e756e6320"
        "706f73756572652e2055742068656e6472657269742073656d7065722076656c"
        "20636c61737320617074656e742074616369746920736f63696f7371752e2041"
        "64206c69746f726120746f727175656e742070657220636f6e75626961206e6f"
        "7374726120696e636570746f732068696d656e61656f732e0a0ab701ffffffff"
        "ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff"
        "ffffffffffffffffffffffffffffffffff4550656f732e0a"
    )
    assert (
        hashlib.sha256(benchmark(lz4.decompress, buf)).hexdigest()
        == "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529"
    )
