from __future__ import annotations

import hashlib
from typing import TYPE_CHECKING

import pytest

from dissect.util.compression import lzxpress

if TYPE_CHECKING:
    from pytest_benchmark.fixture import BenchmarkFixture


@pytest.mark.parametrize(
    ("data", "digest"),
    [
        pytest.param(
            "ffff ff1f 6162 6317 000f ff26 01",
            "d9f5aeb06abebb3be3f38adec9a2e3b94228d52193be923eb4e24c9b56ee0930",
            id="basic",
        ),
        pytest.param(
            "000000004c6f72656d20697073756d20646f6c6f722073697420616d65742063"
            "6f6e7365000000006374657475722061646970697363696e6720656c69742e20"
            "51756973717565200000000066617563696275732065782073617069656e2076"
            "697461652070656c6c656e7401021050652a01731803706c6163657261d80149"
            "6e2069642063757273b8016d69207072657469f8030020006c749001880064d0"
            "02e10376616c6c69732e2054656d70b0006c656f2065752061656e65616e0480"
            "000020736564206469616d2075726e61207410016f722e2050756c76696e6172"
            "1804766100300c026d75732066729005696c6c6120e80378006e656320d80648"
            "00626962656e64756d20656765040200107374613803496163756c6973206d61"
            "737361206e69736c50006c6573756164f201696e100000086961206990066765"
            "72206e756e6320706f73756572652e20557420686802726572690080104474d0"
            "046d70651004656c20636c58022061707448082074616369746920736f63696f"
            "737108c11000752e204164206c69746f72f8056f7271751901c101636f6e7562"
            "50036e6f7374d800696e63ff3f00006570746f732068696d656e61656f732e0a"
            "0ab70d0fff2535",
            "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529",
            id="large",
        ),
    ],
)
def test_lzxpress_decompress(data: str, digest: str) -> None:
    assert hashlib.sha256(lzxpress.decompress(bytes.fromhex(data))).hexdigest() == digest


@pytest.mark.benchmark
def test_benchmark_lzxpress_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex("ffff ff1f 6162 6317 000f ff26 01")
    assert benchmark(lzxpress.decompress, buf) == b"abc" * 100


@pytest.mark.benchmark
def test_benchmark_large_lzxpress_decompress(benchmark: BenchmarkFixture) -> None:
    buf = bytes.fromhex(
        "000000004c6f72656d20697073756d20646f6c6f722073697420616d65742063"
        "6f6e7365000000006374657475722061646970697363696e6720656c69742e20"
        "51756973717565200000000066617563696275732065782073617069656e2076"
        "697461652070656c6c656e7401021050652a01731803706c6163657261d80149"
        "6e2069642063757273b8016d69207072657469f8030020006c749001880064d0"
        "02e10376616c6c69732e2054656d70b0006c656f2065752061656e65616e0480"
        "000020736564206469616d2075726e61207410016f722e2050756c76696e6172"
        "1804766100300c026d75732066729005696c6c6120e80378006e656320d80648"
        "00626962656e64756d20656765040200107374613803496163756c6973206d61"
        "737361206e69736c50006c6573756164f201696e100000086961206990066765"
        "72206e756e6320706f73756572652e20557420686802726572690080104474d0"
        "046d70651004656c20636c58022061707448082074616369746920736f63696f"
        "737108c11000752e204164206c69746f72f8056f7271751901c101636f6e7562"
        "50036e6f7374d800696e63ff3f00006570746f732068696d656e61656f732e0a"
        "0ab70d0fff2535"
    )
    assert (
        hashlib.sha256(benchmark(lzxpress.decompress, buf)).hexdigest()
        == "73d3dd96ca2e2f0144a117019256d770ee7c6febeaee09b24956c723ae22b529"
    )
