"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CredentialsProvider = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const credentials_handler_function_1 = require("./credentials-handler-function");
class CredentialsProvider extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.userArns = [];
        this.secretArns = [];
        // https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-nodejs-readme.html
        const onEvent = new credentials_handler_function_1.CredentialsHandlerFunction(this, "ses-smtp-credentials-handler", {
            timeout: aws_cdk_lib_1.Duration.minutes(1),
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    resources: aws_cdk_lib_1.Lazy.list({
                        produce: () => {
                            return this.userArns;
                        },
                    }),
                    actions: ["iam:CreateAccessKey", "iam:DeleteAccessKey"],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: aws_cdk_lib_1.Lazy.list({
                        produce: () => {
                            return this.secretArns;
                        },
                    }),
                    actions: ["secretsmanager:PutSecretValue"],
                }),
            ],
            logRetention: aws_logs_1.RetentionDays.ONE_MONTH,
        });
        const provider = new custom_resources_1.Provider(this, "ses-smtp-credentials-provider", {
            onEventHandler: onEvent,
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
        });
        this.serviceToken = provider.serviceToken;
    }
    /**
     * Retrieve CredentialsProvider as stack singleton resource.
     *
     * https://github.com/aws/aws-cdk/issues/5023
     */
    static getOrCreate(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = "cdk-ses-smtp-credentials.CredentialsProvider";
        const existing = stack.node.tryFindChild(id);
        return existing || new CredentialsProvider(stack, id);
    }
    grant(user, secret) {
        this.userArns.push(user.userArn);
        this.secretArns.push(secret.secretArn);
    }
}
exports.CredentialsProvider = CredentialsProvider;
//# sourceMappingURL=data:application/json;base64,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