import * as iam from "aws-cdk-lib/aws-iam";
import * as secretsmanager from "aws-cdk-lib/aws-secretsmanager";
import { Construct } from "constructs";
export interface SesSmtpCredentialsProps {
    /**
     * The user for which to create an AWS Access Key and to generate the smtp password. If omitted a user will be created.
     */
    readonly user?: iam.IUser;
    /**
     * Optional, a username to create a new user if no existing user is given.
     */
    readonly userName?: string;
    /**
     * Optional, an SecretsManager secret to write the AWS SES Smtp credentials to.
     */
    readonly secret?: secretsmanager.ISecret;
}
/**
 * This construct creates an access key for the given user and stores the generated SMTP credentials inside a secret.
 *
 * Attaches an inline policy to the user allowing to send emails
 *
 * ```typescript
 * const user = User.fromUserName("ses-user-example");
 * const credentials = new SesSmtpCredentials(this, 'SmtpCredentials', {
 *     user: user,
 * });
 * // smtpCredentials.secret contains json value {username: "<the generated access key id>", password: "<the calculated ses smtp password>"}
 * ```
 */
export declare class SesSmtpCredentials extends Construct {
    /**
     * The secret that contains the calculated AWS SES Smtp Credentials.
     *
     * ```typescript
     * import { aws_ecs } from "aws-cdk-lib";
     *
     * const containerDefinitionOptions: aws_ecs.ContainerDefinitionOptions = {
     *     // ...
     *     secrets: {
     *         MAIL_USERNAME: aws_ecs.Secret.fromSecretsManager(smtpCredentials.secret, "username"),
     *         MAIL_PASSWORD: aws_ecs.Secret.fromSecretsManager(smtpCredentials.secret, "password"),
     *     }
     * }
     * ```
     */
    readonly secret: secretsmanager.ISecret;
    constructor(scope: Construct, id: string, props: SesSmtpCredentialsProps);
}
