from __future__ import annotations

import typing as t
from abc import ABC, abstractmethod
from dataclasses import dataclass, field

from pylox import grammar
from pylox.environment import Environment
from pylox.error import LoxReturnError, LoxRuntimeError
from pylox.protocols.interpreter import SourceInterpreterProtocol
from pylox.tokens import Token, TokenType


class LoxCallable(ABC):  # pragma: no cover
    @property
    @abstractmethod
    def arity(self) -> int:
        raise NotImplementedError

    @abstractmethod
    def call(self, interpreter: SourceInterpreterProtocol, arguments: list[t.Any]) -> t.Any:
        raise NotImplementedError


SelfLoxFunction = t.TypeVar("SelfLoxFunction", bound="LoxFunction")


@dataclass
class LoxFunction(LoxCallable):
    declaration: grammar.Function
    closure: Environment
    is_initializer: bool = False

    def bind(self, instance: LoxInstance) -> LoxFunction:
        """For class methods, define a nested closure with the instance pre-defined as `this`."""
        env = Environment(self.closure)
        env.define(Token(TokenType.THIS, "this", None, 0, 0), instance)
        return LoxFunction(self.declaration, env, self.is_initializer)

    def call(self, interpreter: SourceInterpreterProtocol, arguments: list[t.Any]) -> t.Any:
        """Call the current function instance using the provided arguments."""
        environment = Environment(self.closure)

        for param, val in zip(self.declaration.params, arguments, strict=False):
            environment.define(param, val)

        try:
            interpreter._execute_block(self.declaration.body, environment)
        except LoxReturnError as func_return:
            # Return current instance if we short-circuit from the class init
            if self.is_initializer:
                return self.closure.get_at(0, "this")

            return func_return.value

        # Class constructor will always return the current instance, even if called directly
        if self.is_initializer:
            return self.closure.get_at(0, "this")

    @property
    def arity(self) -> int:
        return len(self.declaration.params)

    def __str__(self) -> str:
        return f"<fn {self.declaration.name.lexeme}>"


@dataclass
class LoxClass(LoxCallable):
    name: str
    superclass: t.Optional[LoxClass]
    methods: dict[str, LoxFunction]

    def call(self, interpreter: SourceInterpreterProtocol, arguments: list[t.Any]) -> LoxInstance:
        instance = LoxInstance(self)

        # Check for an initializer & call it if defined
        initializer = self.find_method("init")
        if initializer is not None:
            initializer.bind(instance).call(interpreter, arguments)

        return instance

    @property
    def arity(self) -> int:
        initializer = self.find_method("init")
        if initializer is None:
            return 0
        else:
            return initializer.arity

    def find_method(self, name: str) -> t.Optional[LoxFunction]:
        """
        Attempt to find a class method with the provided name.

        If the current class is a subclass, locally defined methods overload a superclass method of
        the same name. If the subclass does not overload the method, the inheritance is followed
        upwards to attempt to locate a method.
        """
        if name in self.methods:
            return self.methods[name]

        if self.superclass is not None:
            return self.superclass.find_method(name)

        return None

    def __str__(self) -> str:
        return f"<cls {self.name}>"


@dataclass
class LoxInstance:
    instance_of: LoxClass
    fields: dict = field(default_factory=dict)

    def get(self, name: Token) -> t.Any:
        if name.lexeme in self.fields:
            return self.fields[name.lexeme]

        method = self.instance_of.find_method(name.lexeme)
        if method is not None:
            # bind gives the method a closure with the instance pre-defined as "this"
            return method.bind(self)

        raise LoxRuntimeError(name, f"Undefined property '{name.lexeme}'.")

    def set(self, name: Token, value: t.Any) -> None:
        self.fields[name.lexeme] = value

    def __str__(self) -> str:
        return f"<inst {self.instance_of.name}>"
