# lammps_input_generator.py

def generate_lammps_input(
    units="metal",
    atom_style="atomic",
    read_data="YBCO.data",
    masses=None,
    pair_styles=None,
    pair_coeffs=None,
    run_steps=400
):
    """
    Generates a LAMMPS input file with the specified parameters.
    
    masses: list of tuples [(atom_type, mass), ...]
    pair_styles: list of pair_style strings
    pair_coeffs: list of pair_coeff strings, USE ABSOLUTE PATH FOR POTENTIAL LOCATION
    """
    masses = masses or [(1, 15.999), (2, 63.546), (3, 88.90585), (4, 137.327)]
    pair_styles = pair_styles or ["hybrid/overlay tabgap eam/fs"]
    pair_coeffs = pair_coeffs or [
        "* * tabgap ../Ba-Cu-O-Y.tabgap.h5 O Cu Y Ba no yes",
        "* * eam/fs ../Ba-Cu-O-Y.eam.fs O Cu Y Ba"
    ]

    lines = []
    # Basic setup
    lines.append(f"units {units}")
    lines.append(f"dimension 3")
    lines.append(f"boundary p p p")
    lines.append("")
    lines.append(f"atom_style {atom_style}")
    lines.append("")
    lines.append(f"read_data ../{read_data}")
    lines.append("")

    # Masses
    for atom_type, mass in masses:
        lines.append(f"mass {atom_type} {mass}")
    lines.append("")

    # Pair styles
    for ps in pair_styles:
        lines.append(f"pair_style {ps}")
    lines.append("")
    for pc in pair_coeffs:
        lines.append(f"pair_coeff {pc}")
    lines.append("")

    lines.extend([
        "neigh_modify delay 0 every 1 check yes",
        "",
        f"thermo_style custom step temp pe ke press vol cella cellb cellc",
        f"thermo 10",
        ""
    ])

    # Box and outer shell
    lines.extend([
        "# Define the entire simulation box dimensions",
        "variable xlo equal bound(all,xmin)",
        "variable xhi equal bound(all,xmax)",
        "variable ylo equal bound(all,ymin)",
        "variable yhi equal bound(all,ymax)",
        "variable zlo equal bound(all,zmin)",
        "variable zhi equal bound(all,zmax)",
        "",
        "# Define variables for the outer shell boundaries",
        "variable xlo_outer equal ${xlo} + 2.5",
        "variable xhi_outer equal ${xhi} - 2.5",
        "variable ylo_outer equal ${ylo} + 2.5",
        "variable yhi_outer equal ${yhi} - 2.5",
        "variable zlo_outer equal ${zlo} + 2.5",
        "variable zhi_outer equal ${zhi} - 2.5",
        "",
        "# Define regions for the outer shell",
        "region outer_xlo block ${xlo} ${xlo_outer} INF INF INF INF",
        "region outer_xhi block ${xhi_outer} ${xhi} INF INF INF INF",
        "region outer_ylo block INF INF ${ylo} ${ylo_outer} INF INF",
        "region outer_yhi block INF INF ${yhi_outer} ${yhi} INF INF",
        "region outer_zlo block INF INF INF INF ${zlo} ${zlo_outer}",
        "region outer_zhi block INF INF INF INF ${zhi_outer} ${zhi}",
        "",
        "# Combine the regions for the outer shell",
        "region outer_shell union 6 outer_xlo outer_xhi outer_ylo outer_yhi outer_zlo outer_zhi",
        "",
        "# Group atoms in the outer shell",
        "group boundary region outer_shell",
        "",
        "group core subtract all boundary",
        f"group PKA id MYPKAID",
        "",
        f"velocity PKA set MYPKAVEL",
        "fix core_fix all nve",
        "fix boundary_fix boundary langevin 0 0 0.1 48279",
        f"dump positions all atom {run_steps} prod.data",
        "dump_modify positions first yes",
        "",
        f"run {run_steps}"
    ])

    # Write to file
    with open('tde.in', "w") as f:
        f.write("\n".join(lines))

    print(f"LAMMPS input file 'tde.in' generated successfully!")


# Example usage
if __name__ == "__main__":
    generate_lammps_input()
