from abc import abstractmethod
from typing import Literal

from typing_extensions import override

from pipelex import log
from pipelex.core.memory.exceptions import WorkingMemoryStuffNotFoundError
from pipelex.core.memory.working_memory import WorkingMemory
from pipelex.core.pipes.exceptions import PipeRunInputsError
from pipelex.core.pipes.pipe_abstract import PipeAbstract
from pipelex.core.pipes.pipe_output import PipeOutput
from pipelex.pipe_run.pipe_run_params import PipeRunMode, PipeRunParams
from pipelex.pipeline.job_metadata import JobMetadata


class PipeController(PipeAbstract):
    pipe_category: Literal["PipeController"] = "PipeController"

    @property
    def class_name(self) -> str:
        return self.__class__.__name__

    def _validate_inputs_in_memory(self, working_memory: WorkingMemory) -> None:
        missing_inputs: dict[str, str] = {}
        for required_stuff_name, requirement in self.needed_inputs().items:
            try:
                working_memory.get_stuff(required_stuff_name)
            except WorkingMemoryStuffNotFoundError as exc:
                variable_name: str = exc.variable_name or required_stuff_name
                missing_inputs[variable_name] = exc.concept_code or requirement.concept.code
        if missing_inputs:
            raise PipeRunInputsError(
                message=f"Missing required inputs for pipe '{self.code}': {missing_inputs}", pipe_code=self.code, missing_inputs=missing_inputs
            )

    @override
    async def run_pipe(
        self,
        job_metadata: JobMetadata,
        working_memory: WorkingMemory,
        pipe_run_params: PipeRunParams,
        output_name: str | None = None,
        print_intermediate_outputs: bool | None = False,
    ) -> PipeOutput:
        pipe_run_params.push_pipe_to_stack(pipe_code=self.code)
        self.monitor_pipe_stack(pipe_run_params=pipe_run_params)

        updated_metadata = JobMetadata(
            pipe_job_ids=[self.code],
        )
        job_metadata.update(updated_metadata=updated_metadata)

        # check we have the required inputs in the working memory
        self._validate_inputs_in_memory(working_memory=working_memory)

        pipe_run_info = self._format_pipe_run_info(pipe_run_params=pipe_run_params)
        # log.info(pipe_run_info)
        if pipe_run_params.run_mode == PipeRunMode.LIVE:
            log.info(pipe_run_info)
        match pipe_run_params.run_mode:
            case PipeRunMode.LIVE:
                pipe_output = await self._run_controller_pipe(
                    job_metadata=job_metadata,
                    working_memory=working_memory,
                    pipe_run_params=pipe_run_params,
                    output_name=output_name,
                )
            case PipeRunMode.DRY:
                pipe_output = await self._dry_run_controller_pipe(
                    job_metadata=job_metadata,
                    working_memory=working_memory,
                    pipe_run_params=pipe_run_params,
                    output_name=output_name,
                )

        pipe_run_params.pop_pipe_from_stack(pipe_code=self.code)
        return pipe_output

    @abstractmethod
    async def _run_controller_pipe(
        self,
        job_metadata: JobMetadata,
        working_memory: WorkingMemory,
        pipe_run_params: PipeRunParams,
        output_name: str | None = None,
    ) -> PipeOutput:
        pass

    @abstractmethod
    async def _dry_run_controller_pipe(
        self,
        job_metadata: JobMetadata,
        working_memory: WorkingMemory,
        pipe_run_params: PipeRunParams,
        output_name: str | None = None,
    ) -> PipeOutput:
        pass
