from dataclasses import dataclass
from typing import Annotated, Literal, Optional, List
from pydantic import Field
import math

StructureType = Literal['fixed', 'tracker']
ModuleOrientation = Literal['horizontal', 'vertical']


# ----------------------------- HELPERS ----------------------------------------


def sin_deg(angle_deg: float) -> float:
    """
    The sine of an angle given in degrees.
    """
    angle_rad = math.radians(angle_deg)  # Convert degrees to radians
    return math.sin(angle_rad)


# ----------------------------- PV MODULE ----------------------------------------


@dataclass
class ModuleDesign:
    rated_power: float
    """The power of the module under standard-test conditions (STC): 25°C and 1000
    W/m^2"""

    short_side: float = 1
    """The length of the short side of the module [m]"""

    long_side: float = 2
    """The length of the long side of the module [m]"""

    bifaciality_factor: float = 0
    """A unit-less factor with values between 0 and 1 that describes the ratio of
    the efficiency of the rear side of the module and the efficiency of the front
    side. Bifacial modules typically have a bifaciality factor or 0.7, monofacial
    modules have a bifaciality factor of zero."""

    degradation_rate: float = 0
    """The rate of power reduction over time [%/year]"""

    cell_string_count: int = 3
    """The number of parallel strings into which all cells of the module are
    divided [dimensionless]."""

    half_cell: bool = False
    """True: half-cell module, False: full-cell module."""

    @property
    def module_area(self) -> float:
        """The area of the module [m^2]."""
        return self.short_side * self.long_side


# ----------------------------- INVERTER -----------------------------------------
@dataclass
class InverterDesign:
    rated_power: float
    # TODO: Finalize, review how pvlib implements inverters


# ----------------------------- FIXED-TILT STRUCTURE -----------------------------
@dataclass
class FixedStructureDesign:
    tilt: float
    """The tilt angle of the module surface [degees]. A value of 0 means modules
    are facing up. A Value of 90 means modules are facing the horizon. Values must
    be >= 0 and <= 90."""

    azimuth: float = 180
    """The azimuth angle of the module surface. A value of 0 means modules are facing
    north [degrees] In the northern hemisphere fixed-tilt structures are typically
    oriented towards the south (azimuth = 180 degrees), while in the southern
    hemisphere they are typically oriented towards the north (azimuth = 0 degrees).
    Values must be >= 0 and <= 360.
    """

    clearance: float = 1
    """The shortest distance between any module and the ground [m]. Values must be >= 0"""


# ----------------------------- TRACKER STRUCTURE --------------------------------
@dataclass
class TrackerStructureDesign:
    axis_height: float = 1.5
    """The distance between the axis of rotation and the ground [m]."""

    axis_azimuth: float = 0
    """The angle between the tracker axis and a line oriented toward true north
    [degrees]. Values must be >=0 and <=90. A value of 0 means the tracker axis is
    oriented north-south. A value of 90 means the tracker axis is oriented
    east-west."""

    axis_tilt: float = 0
    """The angle between the axis of rotation and a flat horizontal surface
    [degrees]. Values must be >= 0 and <= 45 degrees."""

    max_tracking_angle: float = 60
    """The maximum possible rotation angle [degrees]. Commercial
    horizontal-single-axis-trackers (HSAT) allow tracking angles up to 50-60
    degrees. Values must be >= 0 and <= 90 degrees."""

    night_stow_angle: float = 0
    """The angle at which the tracker is stowed at night [degrees]. Values must
    be >= 0 and <= 90 degrees."""

    backtracking: bool = True
    """True: backtracking enabled, False: No backtracking"""


# ----------------------------- ARRAY --------------------------------------------
type StructureDesign = FixedStructureDesign | TrackerStructureDesign


@dataclass
class ArrayDesign:
    # --- Components ---

    module: ModuleDesign
    """The PV modules mounted on the structure."""

    structure: StructureDesign
    """Modules are either mounted on a rigid structure with a fixed tilt angle or
    on a Horizontal-Single-Axis-Tracker (HSAT), following the sun from east to west."""

    inverter: InverterDesign
    """The inverter transforming DC to AC current."""

    # --- Arrangement of components ---
    rated_dc_power: float
    """The rated DC power of the array meaning the nominal module power at STC
    conditions multiplied
    with the number of modules [W]. Also called the DC capacity."""

    module_placement: Annotated[str, Field('2v', pattern=r'^\d[vh]$')]  # placement_type
    """A string identifying the arrangement of modules on the structure. For example,
    '2v' indicates two vertically oriented (portrait) modules in the cross-section of
    the structure, with their short sides aligned with the structure's main axis.
    Conversely, '3h' indicates three horizontally oriented (landscape) modules, with
    their long sides aligned with the structure's main axis."""

    dc_ac_ratio: float = 1.2
    """The ratio between the nominal dc and the nominal ac power [fraction]."""

    ground_cover_ratio: float = 0.35
    """The ratio between the collector width and the structure pitch."""

    modules_per_string: int = 28
    """The number of modules per string."""

    modules_per_structure: int = 84
    """The number of modules installed per structure. Typically this number is a
    multiple of the number of modules per string."""

    structures_per_structure_line: int = 1
    """The number of structures connected to lines for efficient robotic cleaning."""

    # --- Ground below PV modules ---

    albedo_value: float = 0.2
    """A single value describing the albedo of the ground below the modules [fraction].
    The default value is 0.2 meaning that 20% of the incoming irradiance is reflected
    (in all directions)."""

    slope_tilt: float = 0
    """The angle of the slope (ground) containing the tracker axes, relative to horizontal
    [degrees]. The default is zero degrees (flat surface)."""

    slope_azimuth: float = 0
    """Direction of the normal to the slope (ground) containing the tracker axes, when
    projected on the horizontal [degrees]. The default is zero degrees (flat surface)."""

    # --- Properties ---

    @property
    def module_count(self) -> float:
        """The number of PV modules belonging to this array."""
        return self.rated_dc_power / self.module.rated_power

    @property
    def rated_ac_power(self) -> float:
        """The nominal ac power of the array meaning the nominal inverter power
        multiplied with the number of inverters [W]. Also called the AC capacity."""
        return self.rated_ac_power / self.dc_ac_ratio

    @property
    def inverter_count(self) -> float:
        """The number of inverters belonging to this array."""
        return self.rated_dc_power / self.inverter.rated_power

    @property
    def string_count(self) -> float:
        """The number of strings belonging to this array."""
        return self.module_count / self.modules_per_string

    @property
    def structure_count(self) -> float:
        """The number of structures belonging to this array."""
        return self.module_count / self.modules_per_structure

    @property
    def total_module_surface_area(self) -> float:  # area_sizing
        """The total module surface area belonging to this array [m^2]."""
        return self.module_count * self.module.module_area

    @property
    def structure_type(self) -> StructureType:
        """The type of structure used: fixed or tracker"""
        if isinstance(self.structure, FixedStructureDesign):
            return 'fixed'
        else:
            return 'tracker'

    @property
    def module_orientation(self) -> ModuleOrientation:
        """The orientation of the long side of the module
        Two options: horizontal, vertical"""
        orientation_char = self.module_placement[1]
        if orientation_char == 'v':
            return 'vertical'
        else:
            return 'horizontal'

    @property
    def number_modules_cross_section(self) -> int:
        """The number of modules in the cross-section of the structure."""
        return int(self.module_placement[0])

    @property
    def collector_width(self) -> float:
        """The width of the rectangle formed by the PV modules placed on top of the structure."""
        if self.module_orientation == 'horizontal':
            return self.number_modules_cross_section * self.module.short_side
        else:  # 'vertical'
            return self.number_modules_cross_section * self.module.long_side

    @property
    def module_clearance(self) -> float:
        """The shortest distance (at any moment of the day) between the lower edge of any PV module and the ground."""
        if isinstance(self.structure, FixedStructureDesign):
            return self.structure.clearance
        else:
            return self.structure.axis_height - 0.5 * self.collector_width * sin_deg(self.structure.max_tracking_angle)

    @property
    def pitch(self) -> float:
        """The distance between the axes of two adjacent structures [m]"""
        return self.collector_width / self.ground_cover_ratio


# ----------------------------- TRANSFORMER --------------------------------------
@dataclass
class TransformerDesign:
    pass


# ----------------------------- GRID ---------------------------------------------
@dataclass
class GridDesign:
    grid_limit: Optional[float] = None


# ----------------------------- SITE ----------------------------------------
@dataclass
class PvradarSiteDesign:
    arrays: List[ArrayDesign]
    transformer: Optional[TransformerDesign]  # ac
    grid: GridDesign

    @property
    def array(self) -> ArrayDesign:
        if not self.arrays:
            raise ValueError('No arrays defined in the design')
        if len(self.arrays) == 1:
            return self.arrays[0]
        else:
            raise NotImplementedError(f'.array property ambiguous, since site design has {len(self.arrays)} arrays')
