"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * SQS queue visibility timeout of Lambda Event Source Mapping is at least 6 times timeout of Lambda function
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_lambda_1.CfnEventSourceMapping) {
        const sourceArn = (0, flatten_cfn_reference_1.flattenCfnReference)(aws_cdk_lib_1.Stack.of(node).resolve(node.eventSourceArn) ?? '');
        if (!sourceArn) {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
        const sourceSqsQueue = getSourceSqsQueue(node, sourceArn);
        if (!sourceSqsQueue) {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
        const queueVisibilityTimeoutSetting = aws_cdk_lib_1.Stack.of(node).resolve(sourceSqsQueue.visibilityTimeout);
        const queueVisibilityTimeout = typeof queueVisibilityTimeoutSetting === 'number' // can be 0, just testing for value truthiness would be wrong
            ? queueVisibilityTimeoutSetting
            : 30; // default SQS Queue visibility timeout
        const lambdaFunctionTimeout = getLambdaFunctionTimeout(node);
        if (!lambdaFunctionTimeout) {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
        if (lambdaFunctionTimeout > queueVisibilityTimeout / 6) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
/**
 * Helper function to get the SQS queue of the Event Source Mapping
 * @param node the CfnEventSourceMapping
 * @param sourceArn the already flattened reference to the source Arn
 * returns the source CfnQueue or undefined if not found
 */
function getSourceSqsQueue(node, sourceArn) {
    for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
        if (child instanceof aws_sqs_1.CfnQueue &&
            (0, flatten_cfn_reference_1.flattenCfnReference)(aws_cdk_lib_1.Stack.of(node).resolve(child.attrArn)) === sourceArn) {
            return child;
        }
    }
    return undefined;
}
/**
 * Helper function to get timeout setting of the CfnEventSourceMapping's Lambda function
 * @param node the CfnEventSourceMapping
 * returns the timeout value of the Lambda function or undefined if not found
 */
function getLambdaFunctionTimeout(node) {
    const functionRef = (0, flatten_cfn_reference_1.flattenCfnReference)(aws_cdk_lib_1.Stack.of(node).resolve(node.functionName));
    for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
        if (child instanceof aws_lambda_1.CfnFunction &&
            (0, flatten_cfn_reference_1.flattenCfnReference)(functionRef) ===
                (0, flatten_cfn_reference_1.flattenCfnReference)(aws_cdk_lib_1.Stack.of(node).resolve(child.ref))) {
            const timeoutSetting = aws_cdk_lib_1.Stack.of(node).resolve(child.timeout);
            if (typeof timeoutSetting === 'number') {
                return timeoutSetting;
            }
            else {
                return 3; // default Lambda function timeout
            }
        }
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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