# Comp-LEO SDK Architecture

This document describes the technical architecture of the Comp-LEO SDK.

## System Overview

```
┌─────────────────────────────────────────────────────────────────┐
│                        User Interfaces                          │
├──────────────────┬──────────────────┬──────────────────────────┤
│  CLI Tool        │  API Service     │  CI/CD Integrations       │
│  comp-leo        │  FastAPI         │  GitHub Actions           │
│                  │  REST endpoints  │  GitLab CI                │
└────────┬─────────┴────────┬─────────┴─────────┬────────────────┘
         │                  │                   │
         └──────────────────┴───────────────────┘
                            │
         ┌──────────────────▼───────────────────┐
         │      Compliance Checker              │
         │  - Orchestrates analysis pipeline    │
         │  - Loads policy packs                │
         │  - Computes scores                   │
         └──────────────────┬───────────────────┘
                            │
         ┌──────────────────▼───────────────────┐
         │      Static Analysis Engine          │
         ├──────────────────────────────────────┤
         │  Leo Parser    →    AST Builder      │
         │  Pattern       →    Scorer           │
         │  Matcher            Evidence         │
         └──────────────────┬───────────────────┘
                            │
         ┌──────────────────▼───────────────────┐
         │      Policy Engine                   │
         ├──────────────────────────────────────┤
         │  Rule Definitions (JSON)             │
         │  - NIST 800-53                       │
         │  - ISO 27001                         │
         │  - PCI-DSS                           │
         │  - GDPR                              │
         │  - Aleo Baseline                     │
         └──────────────────────────────────────┘
```

## Core Components

### 1. Leo Parser (`comp_leo/analyzer/parser.py`)

**Purpose:** Parse Leo source code into analyzable AST structures.

**Key Classes:**
- `LeoParser`: Main parser class
- `LeoProgram`: Represents parsed program
- `LeoFunction`: Function/transition metadata
- `LeoStruct`: Struct definitions
- `LeoMapping`: On-chain state mappings

**Capabilities:**
- Extract program structure without full compilation
- Identify transitions, functions, structs, mappings
- Detect state-mutating operations
- Track visibility (public/private)
- Locate assertions and validations

**Example:**
```python
parser = LeoParser()
program = parser.parse_file("main.leo")

for transition in program.transitions:
    if transition.modifies_state and not transition.has_assertions:
        print(f"Warning: {transition.name} lacks validation")
```

### 2. Compliance Checker (`comp_leo/analyzer/checker.py`)

**Purpose:** Execute compliance rules and generate violations.

**Architecture:**
```
check_file()
    ↓
parse_source()  →  LeoProgram
    ↓
_run_checks()
    ↓
[Rule 1] → violations[]
[Rule 2] → violations[]
[Rule N] → violations[]
    ↓
_compute_score()
    ↓
CheckResult
```

**Check Functions:**
Each rule has a corresponding `_check_<rule_id>()` method:
- `_check_input_validation_missing()`: Detects missing input checks
- `_check_state_mutation_unprotected()`: Finds unprotected state changes
- `_check_logging_insufficient()`: Validates audit trails
- `_check_integer_overflow_risk()`: Detects arithmetic risks
- `_check_private_data_exposure()`: Privacy violations
- `_check_reentrancy_vulnerability()`: Cross-contract call safety
- ... (extensible)

**Scoring Algorithm:**
```python
def _compute_score(violations, total_checks):
    severity_weights = {
        CRITICAL: 20,
        HIGH: 10,
        MEDIUM: 5,
        LOW: 2,
        INFO: 0
    }
    
    penalty = sum(weights[v.severity] for v in violations)
    max_penalty = total_checks * weights[HIGH]
    
    return max(0, 100 - int((penalty / max_penalty) * 100))
```

### 3. Policy Engine (`comp_leo/policies/`)

**Purpose:** Manage compliance frameworks and rule definitions.

**Structure:**
```
policies/
├── rule_engine.py          # Rule loading & management
├── aleo-baseline.json      # Core Leo best practices
├── nist-800-53.json        # Federal security (planned)
├── iso-27001.json          # Info security (planned)
└── pci-dss.json            # Payment card (planned)
```

**Policy Pack Schema:**
```json
{
  "id": "policy-id",
  "name": "Display Name",
  "description": "...",
  "version": "1.0.0",
  "frameworks": ["NIST", "ISO"],
  "enabled_rules": ["rule-1", "rule-2"],
  "fail_threshold": 75,
  "rules": [
    {
      "id": "rule-id",
      "name": "Rule Name",
      "severity": "high|medium|low",
      "violation_type": "security|access_control|...",
      "controls": [
        {
          "framework": "NIST-800-53",
          "control_id": "AC-3",
          "control_name": "Access Enforcement"
        }
      ],
      "enabled": true
    }
  ]
}
```

### 4. Data Models (`comp_leo/core/models.py`)

**Core Models:**

#### Violation
Represents a single compliance issue.
```python
Violation(
    rule_id="input-validation-missing",
    severity=Severity.HIGH,
    message="Transition lacks input validation",
    file_path="main.leo",
    line_number=45,
    remediation=Remediation(
        description="Add assertions",
        code_example="assert(amount > 0u64);"
    ),
    controls=[ControlMapping(...)]
)
```

#### CheckResult
Aggregated results from a compliance scan.
```python
CheckResult(
    passed=True,
    score=85,
    violations=[...],
    total_checks=20,
    critical_count=0,
    high_count=2
)
```

### 5. CLI Tool (`comp_leo/cli/`)

**Commands:**
- `comp-leo check PATH`: Run compliance checks
- `comp-leo report PATH`: Generate audit report
- `comp-leo list-policies`: Show available policy packs

**Output Formatting:**
- Rich terminal UI with colors and formatting
- Grouped violations by severity
- Code snippets and remediation guidance
- Multiple export formats (JSON, HTML, Markdown)

### 6. API Service (`comp_leo/api/`)

**Stack:**
- FastAPI for REST API
- Pydantic for validation
- Mock auth (replace with Redis/DB in production)

**Endpoints:**
- `GET /health`: Health check
- `GET /usage`: API usage stats
- `POST /v1/check`: Run compliance check
- `GET /v1/policies`: List policy packs

**Authentication:**
```
X-API-Key: <api_key>
```

**Rate Limiting:**
- Freemium: 100 checks/month
- Pro: 1,000 checks/month
- Enterprise: Unlimited

## Data Flow

### Local CLI Check

```
1. User runs: comp-leo check programs/

2. CLI loads policy pack from JSON

3. Parser extracts all .leo files

4. For each file:
   a. Parse source → LeoProgram
   b. Apply each enabled rule
   c. Collect violations
   
5. Aggregate results

6. Compute score

7. Format output (Rich terminal or JSON)

8. Exit with code (0 = pass, 1 = fail)
```

### API-Based Check

```
1. Client sends POST /v1/check
   Headers: X-API-Key
   Body: { source, policy_pack }

2. API verifies key & rate limit

3. Write source to temp file

4. ComplianceChecker.check_file()

5. Delete temp file

6. Return CheckResult + usage info

7. Increment usage counter
```

## Extensibility Points

### Adding New Rules

1. **Define rule in policy pack JSON:**
```json
{
  "id": "my-new-rule",
  "name": "My New Rule",
  "severity": "medium",
  "violation_type": "security",
  "enabled": true
}
```

2. **Implement check function in checker.py:**
```python
def _check_my_new_rule(self, program: LeoProgram, rule: Dict) -> List[Violation]:
    violations = []
    
    for transition in program.transitions:
        if <condition>:
            violations.append(self._create_violation(
                rule=rule,
                program=program,
                line=transition.line_start,
                message="...",
                remediation=Remediation(...)
            ))
    
    return violations
```

### Adding New Policy Packs

1. Create `comp_leo/policies/<name>.json`
2. Follow policy pack schema
3. Enable/disable specific rules
4. Set custom threshold

### Parser Extensions

To support new Leo syntax:

1. Add pattern to `LeoParser`
2. Extract to appropriate data structure
3. Update `LeoProgram`, `LeoFunction`, etc.

## Performance Characteristics

### Parsing
- **Speed:** ~1000 lines/second
- **Memory:** O(n) where n = lines of code
- **Bottleneck:** Regex pattern matching

### Rule Evaluation
- **Speed:** ~50 rules/second per file
- **Complexity:** O(rules × functions) typically
- **Optimization:** Early exit on critical violations

### Scaling
- **Single file:** <100ms
- **10 files:** <500ms
- **100 files:** <5s
- **Parallelization:** Future work (process pool)

## Security Considerations

### Local CLI
- ✅ No network calls by default
- ✅ No telemetry
- ✅ Runs offline

### API Service
- 🔒 API key authentication required
- 🔒 Rate limiting per key
- ⚠️ Source code sent to server (use local CLI for sensitive code)
- 🔄 Future: ZK proofs for privacy-preserving checks

## Future Architecture

### Phase 1: AI Auto-Fix

```
ComplianceChecker
    ↓
Violations detected
    ↓
RemediationEngine
    ↓
LLM (GPT-4/Claude)
    ↓
Generate fix patches
    ↓
Confidence scoring
    ↓
[Auto-apply] or [Suggest in PR comment]
```

### Phase 2: Runtime Monitoring

```
Deployed Contract
    ↓
Event Stream
    ↓
Drift Detection Engine
    ↓
Alert on policy violations
```

### Phase 3: Ecosystem Integration

```
Leo Compiler Plugin
    ↓
Pre-compilation checks
    ↓
Block builds on critical issues
```

## Dependencies

**Core:**
- `pydantic`: Data validation
- `click`: CLI framework
- `rich`: Terminal formatting
- `pyyaml`: Config parsing

**Optional (API):**
- `fastapi`: Web framework
- `uvicorn`: ASGI server
- `python-jose`: JWT tokens
- `redis`: Rate limiting (production)

**Development:**
- `pytest`: Testing
- `black`: Code formatting
- `mypy`: Type checking

## Testing Strategy

1. **Unit Tests:** Individual check functions
2. **Integration Tests:** Full check pipeline
3. **Parser Tests:** Leo syntax coverage
4. **API Tests:** Endpoint validation
5. **E2E Tests:** CLI workflows

**Coverage Target:** >80%

## Deployment

### PyPI Package
```bash
python -m build
twine upload dist/*
```

### Docker (API)
```dockerfile
FROM python:3.10-slim
COPY . /app
RUN pip install comp-leo[api]
CMD ["python", "-m", "comp_leo.api.main"]
```

### GitHub Action
Published to GitHub Marketplace as `compiledger/comp-leo-action`

---

**Last Updated:** 2025-10-27  
**Version:** 0.1.0
