# Comp-LEO SDK

**Shift-left compliance & security for Leo smart contracts**

The Comp-LEO SDK brings automated compliance checking, security analysis, and policy enforcement to the Aleo ecosystem. Build safer, audit-ready dApps with real-time feedback during development.

## What It Does

- **Static Analysis**: Parse and analyze Leo code for security vulnerabilities, compliance gaps, and best-practice violations
- **Policy Enforcement**: Built-in rule packs for NIST 800-53, ISO 27001, PCI-DSS, GDPR, and Aleo-specific patterns
- **Developer-First**: Fast local checks (<2s), clear error messages, actionable remediation guidance
- **CI/CD Integration**: Block merges/deployments on policy failures, generate compliance reports
- **Audit Evidence**: Timestamped artifacts mapping code to control IDs for regulatory reviews
- **AI Auto-Fix** (Roadmap): Automated remediation with PR generation

## Quick Start

```bash
# Install CLI
pip install comp-leo

# Check a Leo program
comp-leo check programs/my_contract/src/main.leo

# Run with specific policy pack
comp-leo check --policy pci-dss programs/payment/

# Generate audit report
comp-leo report --format json --output compliance-report.json
```

## Use Cases

### 1. Pre-Commit Hook
```bash
# .git/hooks/pre-commit
#!/bin/bash
comp-leo check --policy nist-800-53 programs/ || exit 1
```

### 2. CI/CD Pipeline
```yaml
# .github/workflows/compliance.yml
- name: Compliance Check
  run: comp-leo check --policy iso-27001 --fail-on-high programs/
```

### 3. API Integration
```python
import requests

response = requests.post(
    "https://api.compiledger.com/v1/check",
    headers={"X-API-Key": "your_key"},
    json={"source": leo_code, "policy": "pci-dss"}
)
```

## Policy Packs

| Pack | Controls | Focus Area |
|------|----------|-----------|
| **nist-800-53** | 1,200+ | Federal security baseline |
| **iso-27001** | 114 | Information security management |
| **pci-dss** | 300+ | Payment card security |
| **gdpr** | 99 | Data protection & privacy |
| **aleo-baseline** | 50+ | Leo-specific best practices |

## Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                      Comp-LEO SDK                           │
├─────────────────────────────────────────────────────────────┤
│  CLI Tool              API Service           CI Integration  │
│  comp-leo check        /v1/check             GitHub Actions  │
│  comp-leo fix          /v1/report            GitLab CI       │
│  comp-leo report       Authentication         PR Comments    │
├─────────────────────────────────────────────────────────────┤
│               Static Analysis Engine                         │
│  Leo Parser → AST → Pattern Matcher → Scorer                │
├─────────────────────────────────────────────────────────────┤
│               Policy Engine                                  │
│  Rules | Severity | Evidence | Control Mapping              │
├─────────────────────────────────────────────────────────────┤
│               Remediation Engine (Future)                    │
│  Fix Generator → AI Agent → PR Creator                       │
└─────────────────────────────────────────────────────────────┘
```

## Example Output

```
⚠️  Compliance Check: 3 issues found

HIGH: Missing input validation [AC-3.1, NIST 800-53]
  → programs/payment/src/main.leo:45
  💡 Add assertion: assert(amount > 0u64);

MEDIUM: Insufficient logging [AU-2, NIST 800-53]
  → programs/payment/src/main.leo:78
  💡 Log transaction hash before state mutation

LOW: Public field exposure [privacy-001]
  → programs/payment/src/main.leo:12
  💡 Consider using private modifier for sensitive data

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
✅ 47 checks passed  ⚠️ 3 warnings  ❌ 0 critical
Score: 85/100 (Threshold: 75)
```

## Pricing

| Tier | Checks/Month | Price | Features |
|------|-------------|-------|----------|
| **Freemium** | 100 | Free | Core policies, CLI access |
| **Pro** | 1,000 | $99/mo | All policies, API access, CI integration |
| **Enterprise** | Unlimited | $999/mo | Custom rules, SLA, white-label |

## Project Structure

```
comp-leo-sdk/
├─ cli/                    # Command-line tool
├─ api/                    # FastAPI service
├─ analyzer/              # Static analysis engine
│  ├─ parser.py           # Leo AST parser
│  ├─ checker.py          # Pattern matcher
│  └─ scorer.py           # Severity & scoring
├─ policies/              # Compliance rule definitions
│  ├─ nist_800_53.json
│  ├─ iso_27001.json
│  ├─ pci_dss.json
│  └─ aleo_baseline.json
├─ integrations/          # CI/CD plugins
│  ├─ github/
│  └─ gitlab/
└─ tests/                 # Test suite
```

## Development Roadmap

### Phase 1: Foundation (Weeks 1-4)
- [x] Leo parser & AST builder
- [x] Core static analysis patterns
- [x] NIST 800-53 baseline (80% of ISO overlap)
- [x] CLI tool with local checks
- [ ] Unit test suite (>80% coverage)

### Phase 2: API & Monetization (Weeks 5-8)
- [ ] FastAPI service with authentication
- [ ] Rate limiting & usage tracking
- [ ] API key management portal
- [ ] Stripe integration for paid tiers

### Phase 3: CI/CD & Ecosystem (Weeks 9-12)
- [ ] GitHub Actions integration
- [ ] PR comment bot
- [ ] Policy pack expansion (PCI, GDPR)
- [ ] VS Code extension

### Phase 4: AI Auto-Fix (Weeks 13-16)
- [ ] Fix suggestion engine
- [ ] LLM integration (GPT-4/Claude)
- [ ] Automated PR generation
- [ ] Confidence scoring for fixes

## Contributing

See [CONTRIBUTING.md](CONTRIBUTING.md) for development setup and guidelines.

## License

Apache 2.0 for core SDK (open-source)
Proprietary for API service & enterprise features

---

**Built for the Aleo ecosystem** | [Website](https://compiledger.com) | [Docs](https://docs.compiledger.com) | [Discord](https://discord.gg/compiledger)
