"""
Comprehensive tests for the schema module.
"""

from pathlib import Path
from provide.testkit.mocking import Mock, patch

import pytest

from plating.schema.processor import SchemaProcessor


class TestSchemaProcessor:
    """Test suite for SchemaProcessor."""

    @pytest.fixture
    def schema_processor(self, mock_generator):
        """Create a SchemaProcessor instance."""
        # Add additional attributes that the schema tests need
        mock_generator.rendered_provider_name = "Test Provider"
        mock_generator.ignore_deprecated = False
        mock_generator.provider_schema = None
        mock_generator.provider_info = None
        mock_generator.resources = {}
        mock_generator.data_sources = {}
        mock_generator.functions = {}
        return SchemaProcessor(mock_generator)

    def test_initialization(self, mock_generator):
        """Test SchemaProcessor initialization."""
        processor = SchemaProcessor(mock_generator)
        assert processor.generator == mock_generator

    def test_extract_provider_schema(self, schema_processor, mock_foundation_hub):
        """Test extract_provider_schema method."""
        # Setup mock schema result
        mock_schema = {
            "provider_schemas": {
                "registry.terraform.io/local/providers/test_provider": {
                    "provider": {"block": {"attributes": {}}},
                    "resource_schemas": {},
                    "data_source_schemas": {},
                    "functions": {},
                }
            }
        }

        # Mock the hub's discovery and component methods
        mock_foundation_hub.discover_components.return_value = None
        mock_foundation_hub.list_components.side_effect = [[], [], [], []]  # For each dimension
        mock_foundation_hub.get_component.return_value = None

        # Replace the processor's hub with our mock
        schema_processor.hub = mock_foundation_hub

        result = schema_processor.extract_provider_schema()

        assert result == mock_schema
        mock_foundation_hub.discover_components.assert_called_once_with("pyvider.components")

    def test_extract_schema_via_discovery(self, schema_processor, mock_foundation_hub, mock_factory):
        """Test _extract_schema_via_discovery method."""
        # Create mock components without get_schema method (will use default)
        mock_provider = mock_factory("provider", spec=[])
        mock_resource = mock_factory("resource", spec=[])
        mock_data = mock_factory("data_source", spec=[])
        mock_func = mock_factory("function", spec=[])

        # Mock the hub's discovery and component retrieval
        mock_foundation_hub.discover_components.return_value = None
        mock_foundation_hub.list_components.side_effect = [
            ["test_provider"],  # provider dimension
            ["test_resource"],  # resource dimension
            ["test_data"],  # data_source dimension
            ["test_func"],  # function dimension
        ]
        mock_foundation_hub.get_component.side_effect = [
            mock_provider,
            mock_resource,
            mock_data,
            mock_func,
        ]

        # Replace the processor's hub with our mock
        schema_processor.hub = mock_foundation_hub

        result = schema_processor._extract_schema_via_discovery()

        assert "provider_schemas" in result
        assert (
            f"registry.terraform.io/local/providers/{schema_processor.generator.provider_name}"
            in result["provider_schemas"]
        )
        mock_foundation_hub.discover_components.assert_called_once_with("pyvider.components")

    def test_get_provider_schema_empty(self, schema_processor):
        """Test _get_provider_schema with empty providers."""
        result = schema_processor._get_provider_schema({})
        assert result == {"block": {"attributes": {}}}

    def test_get_provider_schema_with_schema_method(self, schema_processor):
        """Test _get_provider_schema with component having get_schema method."""
        mock_provider = Mock()
        mock_schema = Mock()
        mock_provider.get_schema.return_value = mock_schema

        with patch("attrs.asdict") as mock_asdict:
            mock_asdict.return_value = {"block": {"attributes": {"test": {}}}}

            schema_processor._get_provider_schema({"provider": mock_provider})

            mock_provider.get_schema.assert_called_once()
            mock_asdict.assert_called_once_with(mock_schema)

    def test_get_component_schemas_empty(self, schema_processor):
        """Test _get_component_schemas with empty components."""
        result = schema_processor._get_component_schemas({})
        assert result == {}

    def test_get_component_schemas_with_get_schema(self, schema_processor):
        """Test _get_component_schemas with components having get_schema method."""
        mock_component = Mock()
        mock_schema = Mock()
        mock_component.get_schema.return_value = mock_schema

        with patch("attrs.asdict") as mock_asdict:
            mock_asdict.return_value = {"block": {"attributes": {}}}

            result = schema_processor._get_component_schemas({"test": mock_component})

            assert "test" in result
            mock_component.get_schema.assert_called_once()

    def test_get_component_schemas_with_pyvider_schema(self, schema_processor):
        """Test _get_component_schemas with __pyvider_schema__ attribute."""
        mock_component = Mock()
        mock_component.__pyvider_schema__ = {"block": {"attributes": {}}}
        del mock_component.get_schema  # Ensure it doesn't have get_schema

        result = schema_processor._get_component_schemas({"test": mock_component})

        assert "test" in result
        assert result["test"] == mock_component.__pyvider_schema__

    def test_get_function_schemas_empty(self, schema_processor):
        """Test _get_function_schemas with empty functions."""
        result = schema_processor._get_function_schemas({})
        assert result == {}

    def test_get_function_schemas_with_get_schema(self, schema_processor):
        """Test _get_function_schemas with functions having get_schema method."""
        mock_func = Mock()
        mock_schema = Mock()
        mock_func.get_schema.return_value = mock_schema

        with patch("attrs.asdict") as mock_asdict:
            mock_asdict.return_value = {"signature": {}}

            result = schema_processor._get_function_schemas({"test_func": mock_func})

            assert "test_func" in result
            mock_func.get_schema.assert_called_once()

    def test_parse_function_signature_basic(self, schema_processor):
        """Test _parse_function_signature with basic function."""
        func_schema = {
            "signature": {
                "parameters": [{"name": "input", "type": "string"}, {"name": "count", "type": "number"}],
                "return_type": "list(string)",
            }
        }

        result = schema_processor._parse_function_signature(func_schema)
        assert result == "function(input: string, count: number) -> list(string)"

    def test_parse_function_signature_with_variadic(self, schema_processor):
        """Test _parse_function_signature with variadic parameter."""
        func_schema = {
            "signature": {
                "parameters": [{"name": "first", "type": "string"}],
                "variadic_parameter": {"name": "rest", "type": "string"},
                "return_type": "string",
            }
        }

        result = schema_processor._parse_function_signature(func_schema)
        assert result == "function(first: string, ...rest: string) -> string"

    def test_parse_function_signature_no_signature(self, schema_processor):
        """Test _parse_function_signature with no signature."""
        result = schema_processor._parse_function_signature({})
        assert result == ""

    def test_parse_function_arguments(self, schema_processor):
        """Test _parse_function_arguments."""
        func_schema = {
            "signature": {
                "parameters": [
                    {"name": "input", "type": "string", "description": "Input value"},
                    {"name": "count", "type": "number", "description": "Number of items"},
                ]
            }
        }

        result = schema_processor._parse_function_arguments(func_schema)
        assert "- `input` (string) - Input value" in result
        assert "- `count` (number) - Number of items" in result

    def test_parse_variadic_argument(self, schema_processor):
        """Test _parse_variadic_argument."""
        func_schema = {
            "signature": {
                "variadic_parameter": {
                    "name": "values",
                    "type": "string",
                    "description": "Variable number of string values",
                }
            }
        }

        result = schema_processor._parse_variadic_argument(func_schema)
        assert result == "- `values` (string) - Variable number of string values"

    def test_parse_provider_schema(self, mock_generator):
        """Test parse_provider_schema method."""
        processor = SchemaProcessor(mock_generator)

        # Setup provider schema
        mock_generator.provider_schema = {
            "provider_schemas": {
                f"registry.terraform.io/local/providers/{mock_generator.provider_name}": {
                    "provider": {"description": "Test provider"},
                    "resource_schemas": {
                        "test_resource": {
                            "description": "Test resource",
                            "block": {"attributes": {"id": {"type": "string", "computed": True}}},
                        }
                    },
                    "data_source_schemas": {
                        "test_data": {
                            "description": "Test data source",
                            "block": {"attributes": {"name": {"type": "string", "required": True}}},
                        }
                    },
                    "functions": {
                        "test_func": {
                            "description": "Test function",
                            "signature": {"parameters": [], "return_type": "string"},
                        }
                    },
                }
            }
        }

        processor.parse_provider_schema()

        # Check provider info was created
        assert mock_generator.provider_info is not None
        assert mock_generator.provider_info.name == mock_generator.provider_name

        # Check resources were parsed
        assert "test_resource" in mock_generator.resources
        assert mock_generator.resources["test_resource"].name == "test_resource"

        # Check data sources were parsed
        assert "test_data" in mock_generator.data_sources
        assert mock_generator.data_sources["test_data"].name == "test_data"

        # Check functions were parsed
        assert "test_func" in mock_generator.functions
        assert mock_generator.functions["test_func"].name == "test_func"

    def test_parse_provider_schema_ignore_deprecated(self, mock_generator):
        """Test parse_provider_schema ignores deprecated resources when flag is set."""
        processor = SchemaProcessor(mock_generator)
        mock_generator.ignore_deprecated = True

        mock_generator.provider_schema = {
            "provider_schemas": {
                f"registry.terraform.io/local/providers/{mock_generator.provider_name}": {
                    "provider": {},
                    "resource_schemas": {
                        "test_resource": {"deprecated": True, "block": {"attributes": {}}},
                        "valid_resource": {"block": {"attributes": {}}},
                    },
                    "data_source_schemas": {},
                    "functions": {},
                }
            }
        }

        processor.parse_provider_schema()

        # Deprecated resource should not be included
        assert "test_resource" not in mock_generator.resources
        assert "valid_resource" in mock_generator.resources

    def test_parse_schema_to_markdown_basic(self, schema_processor):
        """Test _parse_schema_to_markdown with basic schema."""
        schema = {
            "block": {
                "attributes": {
                    "id": {"type": "string", "description": "The ID", "computed": True},
                    "name": {"type": "string", "description": "The name", "required": True},
                    "tags": {"type": {"map": "string"}, "description": "Tags", "optional": True},
                }
            }
        }

        result = schema_processor._parse_schema_to_markdown(schema)

        # Check markdown table format
        assert "| Argument | Type | Required | Description |" in result
        assert "| `id` | String | No (Computed) | The ID |" in result
        assert "| `name` | String | **Yes** | The name |" in result
        assert "| `tags` | String | No | Tags |" in result

    def test_parse_schema_to_markdown_with_blocks(self, schema_processor):
        """Test _parse_schema_to_markdown with nested blocks."""
        schema = {
            "block": {
                "attributes": {},
                "block_types": {
                    "config": {
                        "description": "Configuration block",
                        "block": {
                            "attributes": {
                                "enabled": {"type": "bool", "description": "Enable feature", "optional": True}
                            }
                        },
                    }
                },
            }
        }

        result = schema_processor._parse_schema_to_markdown(schema)

        # Check for nested block header and table
        assert "### config Block" in result
        assert "| Argument | Type | Required | Description |" in result
        assert "| `enabled` | Boolean | No | Enable feature |" in result

    def test_parse_schema_to_markdown_empty(self, schema_processor):
        """Test _parse_schema_to_markdown with empty schema."""
        assert schema_processor._parse_schema_to_markdown({}) == "No arguments available."
        assert schema_processor._parse_schema_to_markdown({"block": {}}) == "No arguments available."

    def test_format_type_string_simple(self, schema_processor):
        """Test _format_type_string with simple types."""
        assert schema_processor._format_type_string("string") == "String"
        assert schema_processor._format_type_string("number") == "Number"
        assert schema_processor._format_type_string("bool") == "Boolean"
        assert schema_processor._format_type_string(None) == "String"

    def test_format_type_string_with_dict(self, schema_processor):
        """Test _format_type_string with dict types."""
        assert schema_processor._format_type_string({}) == "String"
        assert schema_processor._format_type_string({"type": "string"}) == "String"

    @patch("subprocess.run")
    @patch("shutil.rmtree")
    @patch("pathlib.Path.write_text")
    @patch("pathlib.Path.mkdir")
    def test_extract_schema_via_terraform(
        self, mock_mkdir, mock_write_text, mock_rmtree, mock_run, schema_processor
    ):
        """Test _extract_schema_via_terraform fallback method."""
        # Setup mock subprocess returns
        mock_run.side_effect = [
            Mock(returncode=0),  # build command
            Mock(returncode=0),  # terraform init
            Mock(returncode=0, stdout='{"provider_schemas": {}}'),  # terraform schema
        ]

        # Mock finding the binary
        with patch.object(schema_processor, "_find_provider_binary") as mock_find:
            mock_find.return_value = Path("/test/provider/binary")

            result = schema_processor._extract_schema_via_terraform()

            assert result == {"provider_schemas": {}}
            assert mock_run.call_count == 3
            mock_rmtree.assert_called_once()

    @patch("pathlib.Path.glob")
    def test_find_provider_binary(self, mock_glob, schema_processor):
        """Test _find_provider_binary method."""
        # Mock Path.glob to return a list with one matching file
        mock_glob.return_value = [Path("/test/provider/terraform-provider-test")]

        result = schema_processor._find_provider_binary()

        assert result == Path("/test/provider/terraform-provider-test")

    @patch("pathlib.Path.glob")
    def test_find_provider_binary_not_found(self, mock_glob, schema_processor):
        """Test _find_provider_binary raises when binary not found."""
        mock_glob.return_value = []

        with pytest.raises(FileNotFoundError):
            schema_processor._find_provider_binary()


class TestSchemaProcessorWithCTY:
    """Test SchemaProcessor with CTY types."""

    @pytest.fixture
    def schema_processor(self):
        """Create a SchemaProcessor instance."""
        mock_generator = Mock(name="DocsGenerator")
        mock_generator.provider_name = "test"
        mock_generator.provider_dir = Path("/test/provider")
        mock_generator.resources = {}
        mock_generator.data_sources = {}
        mock_generator.functions = {}
        mock_generator.provider_info = None
        return SchemaProcessor(mock_generator)

    @pytest.mark.skip(reason="pyvider.cty is an optional dependency and patching doesn't work correctly")
    @patch("pyvider.cty.CtyString")
    @patch("pyvider.cty.CtyNumber")
    @patch("pyvider.cty.CtyBool")
    def test_format_type_string_with_cty(self, MockCtyBool, MockCtyNumber, MockCtyString, schema_processor):
        """Test _format_type_string with CTY objects."""
        # Test string type
        mock_string = Mock()
        mock_string.__class__ = MockCtyString
        assert schema_processor._format_type_string(mock_string) == "String"

        # Test number type
        mock_number = Mock()
        mock_number.__class__ = MockCtyNumber
        assert schema_processor._format_type_string(mock_number) == "Number"

        # Test bool type
        mock_bool = Mock()
        mock_bool.__class__ = MockCtyBool
        assert schema_processor._format_type_string(mock_bool) == "Boolean"

    @pytest.mark.skip(reason="pyvider.cty is an optional dependency and patching doesn't work correctly")
    def test_format_type_string_with_cty_list(self, schema_processor):
        """Test _format_type_string with CTY list type."""
        # Since the CTY types are imported inside the function, we need to mock the import
        with patch("pyvider.cty.CtyList") as MockCtyList:
            # Create a mock list object
            mock_list = Mock()
            mock_list.element_type = Mock()  # Mock element type

            # Make the mock's class the patched CtyList
            type(mock_list).__name__ = "CtyList"

            # We need to make the comparison work
            MockCtyList.__eq__ = lambda self, other: other.__class__.__name__ == "CtyList"

            # But since the comparison happens inside the function with locally imported CtyList,
            # we can't easily mock it. Let's test with a simpler approach:
            # Test that it handles unknown types gracefully
            result = schema_processor._format_type_string(mock_list)
            # Since CtyList won't be recognized without pyvider.cty installed, it returns String
            assert result == "String"
