# Copyright (c) 2023 - 2025 Chair for Design Automation, TUM
# Copyright (c) 2025 Munich Quantum Software Company GmbH
# All rights reserved.
#
# SPDX-License-Identifier: MIT
#
# Licensed under the MIT License

from __future__ import annotations

from typing import TYPE_CHECKING

import numpy as np

from ..components.extensions.gate_types import GateTypes
from ..gate import Gate

if TYPE_CHECKING:
    from numpy.typing import NDArray

    from ..circuit import QuantumCircuit
    from ..components.extensions.controls import ControlData
    from ..gate import Parameter


class CustomTwo(Gate):
    """Two body custom gate."""

    def __init__(
        self,
        circuit: QuantumCircuit,
        name: str,
        target_qudits: list[int],
        parameters: NDArray[np.complex128],
        dimensions: list[int],
        controls: ControlData | None = None,
    ) -> None:
        super().__init__(
            circuit=circuit,
            name=name,
            gate_type=GateTypes.TWO,
            target_qudits=target_qudits,
            dimensions=dimensions,
            control_set=controls,
            params=parameters,
            qasm_tag="cutwo",
        )
        self.__array_storage: NDArray[np.complex128] | None = None
        if self.validate_parameter(parameters):
            self.__array_storage = parameters

    def __array__(self) -> NDArray[np.complex128]:  # noqa: PLW3201
        if self.__array_storage is None:
            msg = "The gate does not have a matrix assigned."
            raise ValueError(msg)

        return self.__array_storage

    @staticmethod
    def validate_parameter(parameter: Parameter | None = None) -> bool:
        if parameter is None:
            return True  # or False, depending on whether None is considered valid
        return bool(
            isinstance(parameter, np.ndarray)
            and (parameter.dtype == np.complex128 or np.issubdtype(parameter.dtype, np.number))
        )
