#
# BSD 3-Clause License
#
# Copyright (c) 2022, California Institute of Technology and
# Max Planck Institute for Gravitational Physics (Albert Einstein Institute)
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
#    list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# 3. Neither the name of the copyright holder nor the names of its
#    contributors may be used to endorse or promote products derived from
#    this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# This software may be subject to U.S. export control laws. By accepting this
# software, the user agrees to comply with all applicable U.S. export laws and
# regulations. User has the responsibility to obtain export licenses, or other
# export authority as may be required before exporting such information to
# foreign countries or providing access to foreign persons.
#
"""
Test clock-noise correction.
"""

from pytdi import LISAClockCorrection, LISATDICombination
from pytdi.naming import lisa_indices


def test_empty_combination_from_combination():
    """Test `LISAClockCorrection` for an empty combination."""
    combination = LISATDICombination({})
    correction = LISAClockCorrection(combination)
    for i, j in lisa_indices():
        assert correction.P_component[f"{i}{j}"] == []
        assert correction.P[f"{i}{j}"] == LISATDICombination({})
        assert correction.R[f"{i}{j}"] == LISATDICombination({})


def test_empty_combination_from_string():
    """Test `LISAClockCorrection` for an empty combination."""
    combination = LISATDICombination.from_string("")
    correction = LISAClockCorrection(combination)
    for i, j in lisa_indices():
        assert correction.P_component[f"{i}{j}"] == []
        assert correction.P[f"{i}{j}"] == LISATDICombination({})
        assert correction.R[f"{i}{j}"] == LISATDICombination({})


def test_michelson_combination_from_string():
    """Test `LISAClockCorrection` for X1 Michelson combination"""
    combination = LISATDICombination.from_string("12131 -12131")
    correction = LISAClockCorrection(combination)
    for i, j in lisa_indices():
        assert correction.P_component[
            f"{i}{j}"
        ] == correction.combination.components.get(f"eta_{i}{j}", [])
    assert correction.is_valid()


def test_sagnac_combination_from_string():
    """Test `LISAClockCorrection` for alpha1 Sagnac combination, for which the algorithm is invalid"""
    combination = LISATDICombination.from_string("1231 -1231")
    correction = LISAClockCorrection(combination)
    assert not correction.is_valid()
