"""OpenRouter DSPy integration for data4ai."""

import logging
import os
from typing import Any, Optional

import dspy
import httpx
from ratelimit import limits

logger = logging.getLogger(__name__)

# Rate limiting configuration
RL_CALLS = 40
RL_PERIOD_SECONDS = 60


class OpenRouterDSPyClient(dspy.LM):
    """OpenRouter client for DSPy integration."""

    def __init__(
        self,
        api_key: Optional[str] = None,
        base_url: str = "https://openrouter.ai/api/v1",
        model: str = "openai/gpt-4o-mini",
        extra_headers: Optional[dict[str, str]] = None,
        site_url: str = "https://github.com/zysec-ai/data4ai",
        site_name: str = "Data4AI",
        **kwargs,
    ):
        """Initialize OpenRouter DSPy client."""
        self.api_key = api_key or os.getenv("OPENROUTER_API_KEY")
        if not self.api_key:
            raise ValueError("OpenRouter API key is required")

        self.base_url = base_url
        self.model = model
        self.extra_headers = extra_headers or {}
        self.site_url = site_url
        self.site_name = site_name

        # DSPy LM configuration
        self.history = []
        self.provider = "openai"
        self.model = model  # DSPy expects this attribute

        # Default parameters
        self.kwargs = {
            "temperature": 0.7,
            "max_tokens": 2048,
            "top_p": 1,
            "frequency_penalty": 0,
            "presence_penalty": 0,
            "n": 1,
        }
        self.kwargs.update(kwargs)

    def _get_headers(self) -> dict[str, str]:
        """Get headers with proper attribution for analytics."""
        headers = {
            "Authorization": f"Bearer {self.api_key}",
            "Content-Type": "application/json",
            "HTTP-Referer": self.site_url,
            "X-Title": self.site_name,
        }
        headers.update(self.extra_headers)
        return headers

    @limits(calls=RL_CALLS, period=RL_PERIOD_SECONDS)
    def basic_request(self, prompt: str, **kwargs) -> dict[str, Any]:
        """Make a basic request to OpenRouter API."""
        headers = self._get_headers()

        # Merge default kwargs with provided kwargs
        request_kwargs = self.kwargs.copy()
        request_kwargs.update(kwargs)

        data = {
            "model": self.model,
            "messages": [{"role": "user", "content": prompt}],
            **request_kwargs,
        }

        try:
            with httpx.Client(timeout=30) as client:
                response = client.post(
                    f"{self.base_url}/chat/completions", headers=headers, json=data
                )
                response.raise_for_status()
                response_data = response.json()

                # Log successful response
                logger.debug("OpenRouter API response received")

                # Store in history
                self.history.append(
                    {
                        "prompt": prompt,
                        "response": response_data,
                        "kwargs": kwargs,
                    }
                )

                return response_data

        except httpx.HTTPError as e:
            logger.error(f"OpenRouter API request failed: {e}")
            raise
        except Exception as e:
            logger.error(f"Unexpected error in OpenRouter request: {e}")
            raise

    def __call__(self, prompt: str = None, **kwargs) -> list[str]:
        """DSPy LM interface - call the model with a prompt."""
        try:
            # Handle case where prompt might be passed as a keyword argument
            if prompt is None:
                # DSPy passes the prompt as 'messages' in kwargs
                if "messages" in kwargs:
                    # Extract prompt from messages
                    messages = kwargs["messages"]
                    if isinstance(messages, list) and len(messages) > 0:
                        # Get the last user message as the prompt
                        for msg in reversed(messages):
                            if msg.get("role") == "user":
                                prompt = msg.get("content", "")
                                break
                        if not prompt:
                            # Fallback to last message content
                            prompt = messages[-1].get("content", "")
                    else:
                        prompt = str(messages)
                elif "prompt" in kwargs:
                    prompt = kwargs["prompt"]
                else:
                    # Try to find any string argument that could be the prompt
                    for _key, value in kwargs.items():
                        if isinstance(value, str) and len(value) > 10:
                            prompt = value
                            break

                    if prompt is None:
                        raise ValueError("No prompt found in arguments")

            response_data = self.basic_request(prompt, **kwargs)

            # Extract completions from response
            completions = []
            for choice in response_data.get("choices", []):
                if "message" in choice and "content" in choice["message"]:
                    completions.append(choice["message"]["content"])

            return completions

        except Exception as e:
            logger.error(f"Error in DSPy OpenRouter call: {e}")
            raise


def configure_dspy_with_openrouter(
    model: str = "openai/gpt-4o-mini",
    api_key: Optional[str] = None,
    site_url: str = "https://github.com/zysec-ai/data4ai",
    site_name: str = "Data4AI",
    **kwargs,
) -> None:
    """Configure DSPy to use OpenRouter."""
    try:
        client = OpenRouterDSPyClient(
            api_key=api_key,
            model=model,
            site_url=site_url,
            site_name=site_name,
            **kwargs,
        )

        dspy.configure(lm=client)
        logger.debug(f"DSPy configured with OpenRouter model: {model}")

    except Exception as e:
        logger.error(f"Failed to configure DSPy with OpenRouter: {e}")
        raise


class OpenRouterPromptOptimizer:
    """DSPy-based prompt optimizer using OpenRouter."""

    def __init__(
        self,
        model: str = "openai/gpt-4o-mini",
        api_key: Optional[str] = None,
        **kwargs,
    ):
        """Initialize the prompt optimizer."""
        self.model = model
        self.api_key = api_key

        # Add optimizer attribute for detection
        self.optimizer = True  # Indicates this is a DSPy optimizer
        self.use_dspy = True

        # Configure DSPy with OpenRouter
        configure_dspy_with_openrouter(model=model, api_key=api_key, **kwargs)

        self._setup_signatures()

    def _setup_signatures(self):
        """Setup DSPy signatures for dataset generation."""

        class DatasetGenerationSignature(dspy.Signature):
            """DSPy signature for dataset generation."""

            description = dspy.InputField(
                desc="Natural language description of the dataset to generate"
            )
            schema_name = dspy.InputField(desc="Dataset schema name (chatml, alpaca)")
            count = dspy.InputField(desc="Number of examples to generate")
            examples = dspy.OutputField(
                desc="List of high-quality dataset examples in JSON format"
            )

        self.signature = DatasetGenerationSignature

    def generate_dynamic_prompt(
        self,
        description: str,
        schema_name: str,
        count: int,
        examples: Optional[list[dict[str, Any]]] = None,
    ) -> str:
        """Generate a dynamic prompt using DSPy."""
        try:
            # Create a prompt generation signature for DSPy
            class PromptGenerationSignature(dspy.Signature):
                """DSPy signature for generating optimized prompts."""

                description = dspy.InputField(
                    desc="Natural language description of the dataset to generate"
                )
                schema_name = dspy.InputField(
                    desc="Dataset schema name (chatml, alpaca)"
                )
                count = dspy.InputField(desc="Number of examples to generate")
                optimized_prompt = dspy.OutputField(
                    desc="An optimized prompt that will generate high-quality examples"
                )

            # Create predictor with the prompt generation signature
            predictor = dspy.Predict(PromptGenerationSignature)

            logger.debug(f"Generating optimized prompt for {schema_name}")

            # Generate the optimized prompt
            result = predictor(
                description=description, schema_name=schema_name, count=count
            )

            # Check if we got a valid result
            if not hasattr(result, "optimized_prompt") or not result.optimized_prompt:
                logger.warning(
                    "DSPy returned empty result, falling back to static prompt"
                )
                return self._fallback_prompt(description, schema_name, count)

            # Combine DSPy-generated insight with our structured template
            dspy_insight = result.optimized_prompt
            logger.debug("DSPy generated insight successfully")

            structured_prompt = self._enhance_with_dspy_insight(
                dspy_insight, description, schema_name, count
            )

            logger.debug("DSPy prompt optimization completed successfully")
            return structured_prompt

        except Exception as e:
            logger.warning(f"DSPy prompt generation failed: {str(e)}")
            logger.debug("DSPy error details", exc_info=True)
            return self._fallback_prompt(description, schema_name, count)

    def generate_schema_prompt(
        self,
        description: str,
        schema_name: str,
        count: int,
        use_dspy: bool = True,
    ) -> str:
        """Generate a schema-aware prompt (compatibility method)."""
        if use_dspy:
            return self.generate_dynamic_prompt(description, schema_name, count)
        else:
            return self._fallback_prompt(description, schema_name, count)

    def generate_adaptive_prompt(
        self,
        description: str,
        schema_name: str,
        count: int,
        previous_examples: Optional[list[dict[str, Any]]] = None,
    ) -> str:
        """Generate an adaptive prompt (compatibility method)."""
        if previous_examples and len(previous_examples) > 0:
            # For now, use the same logic as generate_dynamic_prompt
            # In the future, this could be enhanced to use previous examples
            return self.generate_dynamic_prompt(description, schema_name, count)
        else:
            return self.generate_schema_prompt(description, schema_name, count)

    def _enhance_with_dspy_insight(
        self, dspy_insight: str, description: str, schema_name: str, count: int
    ) -> str:
        """Enhance DSPy-generated insights with our structured template."""

        # Schema-specific task descriptions

        enhanced_prompt = f"""{dspy_insight}

CRITICAL REQUIREMENTS:
1. You MUST respond with ONLY a valid JSON array
2. Do NOT include any explanatory text, code examples, or other content
3. Do NOT use markdown formatting
4. The response must be parseable JSON
5. Each example must be realistic and useful for training
6. Generate the requested number of high-quality examples
7. Apply the DSPy optimization insights above to improve quality

For {schema_name} schema, each example should have:"""

        if schema_name == "chatml":
            enhanced_prompt += """
- messages: List of conversation messages with role and content (REQUIRED - cannot be empty)

CRITICAL: The 'messages' array MUST contain at least 2 messages (user + assistant). NEVER return empty messages arrays.

Example format:
[
  {{
    "messages": [
      {{"role": "user", "content": "What is 2+2?"}},
      {{"role": "assistant", "content": "2+2 equals 4."}}
    ]
  }},
  {{
    "messages": [
      {{"role": "system", "content": "You are a helpful assistant."}},
      {{"role": "user", "content": "Translate 'hello' to French"}},
      {{"role": "assistant", "content": "Hello in French is 'bonjour'."}}
    ]
  }}
]"""
        elif schema_name == "alpaca":
            enhanced_prompt += """
- instruction: The task or question (REQUIRED - cannot be empty)
- input: Additional context (can be empty string "")
- output: The expected response or answer (REQUIRED - cannot be empty)

Example format:
[
  {{
    "instruction": "Translate the following text to French",
    "input": "Hello, how are you?",
    "output": "Bonjour, comment allez-vous?"
  }},
  {{
    "instruction": "Write a Python function to calculate factorial",
    "input": "",
    "output": "def factorial(n):\\n    if n <= 1:\\n        return 1\\n    return n * factorial(n - 1)"
  }}
]"""

        # Schema-specific response format instructions
        if schema_name == "chatml":
            enhanced_prompt += """

RESPONSE FORMAT: Return ONLY a JSON array with the requested number of examples. No other text.

CRITICAL: Each 'messages' array MUST contain at least 2 messages (user + assistant). NEVER return empty messages arrays.

Remember: Apply the DSPy insights above and return ONLY the JSON array, nothing else."""
        else:
            enhanced_prompt += """

RESPONSE FORMAT: Return ONLY a JSON array with the requested number of examples. No other text.

IMPORTANT: instruction and output must contain meaningful content (never empty). Only input can be empty ("") when not needed.

Remember: Apply the DSPy insights above and return ONLY the JSON array, nothing else."""

        return enhanced_prompt

    def _fallback_prompt(self, description: str, schema_name: str, count: int) -> str:
        """Fallback to static prompt if DSPy fails."""

        # Schema-specific task descriptions
        task_descriptions = {
            "chatml": "conversation examples",
            "alpaca": "instruction-tuning examples",
        }

        task_desc = task_descriptions.get(schema_name, "examples")

        base_prompt = f"""You are a dataset generator. Your task is to create {task_desc} for a {schema_name} dataset.

DESCRIPTION: {description}

CRITICAL REQUIREMENTS:
1. You MUST respond with ONLY a valid JSON array
2. Do NOT include any explanatory text, code examples, or other content
3. Do NOT use markdown formatting
4. The response must be parseable JSON
5. Each example must be realistic and useful for training
6. Generate high-quality examples (exact count will be specified in the user message)

For {schema_name} schema, each example should have:"""

        if schema_name == "chatml":
            base_prompt += """
- messages: List of messages with role and content (REQUIRED - cannot be empty)

CRITICAL: The 'messages' array MUST contain at least 2 messages (user + assistant). NEVER return empty messages arrays.

Example format:
[
  {{
    "messages": [
      {{"role": "user", "content": "What is 2+2?"}},
      {{"role": "assistant", "content": "2+2 equals 4."}}
    ]
  }},
  {{
    "messages": [
      {{"role": "system", "content": "You are a helpful assistant."}},
      {{"role": "user", "content": "Translate 'hello' to French"}},
      {{"role": "assistant", "content": "Hello in French is 'bonjour'."}}
    ]
  }}
]"""
        elif schema_name == "alpaca":
            base_prompt += """
- instruction: The task or question (REQUIRED - cannot be empty)
- input: Additional context (can be empty string "")
- output: The expected response or answer (REQUIRED - cannot be empty)

Example format:
[
  {{
    "instruction": "Translate the following text to French",
    "input": "Hello, how are you?",
    "output": "Bonjour, comment allez-vous?"
  }},
  {{
    "instruction": "Write a Python function to calculate factorial",
    "input": "",
    "output": "def factorial(n):\\n    if n <= 1:\\n        return 1\\n    return n * factorial(n - 1)"
  }}
]"""
        else:
            base_prompt += """
- Follow the custom schema format"""

        base_prompt += """

RESPONSE FORMAT: Return ONLY a JSON array with the requested number of examples. No other text.

Example response format:
[
  {{
    "instruction": "What is 2+2?",
    "input": "Basic arithmetic question",
    "output": "2+2 equals 4."
  }},
  {{
    "instruction": "Translate hello to Spanish",
    "input": "English word to translate",
    "output": "hola"
  }}
]

IMPORTANT: instruction and output must contain meaningful content (never empty). Only input can be empty ("") when not needed.

Remember: Return ONLY the JSON array, nothing else."""

        return base_prompt


# Factory function for easy integration
def create_openrouter_prompt_generator(
    model: str = "openai/gpt-4o-mini",
    api_key: Optional[str] = None,
    **kwargs,
) -> OpenRouterPromptOptimizer:
    """Create an OpenRouter-based prompt generator."""
    return OpenRouterPromptOptimizer(model=model, api_key=api_key, **kwargs)
