# Must CDK Examples

This directory contains comprehensive examples demonstrating the usage of Must CDK constructs in both TypeScript and Python. Each example showcases advanced configurations and best practices for production deployments.

## 📁 Example Structure

### Amplify Examples
- **amplify-advanced.ts** / **amplify_advanced.py**
  - Multi-environment branch configurations
  - Custom domain setup with SSL
  - Advanced build specifications for Next.js
  - Environment-specific variables
  - Pull request previews and auto-deployment

### ECS CodeDeploy Examples
- **ecs-codedeploy.ts** / **ecs_codedeploy.py**
  - Blue/Green deployment strategies
  - Auto-scaling configurations
  - Load balancer integration
  - Health checks and monitoring
  - Security group configurations

### CloudFront Examples
- **cloudfront-distribution.ts** / **cloudfront_distribution.py**
  - Multi-origin configurations (S3 + API Gateway)
  - Custom cache behaviors
  - Security headers and WAF integration
  - Geographic restrictions
  - Custom error pages and logging

### API Gateway Lambda Examples
- **api-gateway-lambda.ts** / **api_gateway_lambda.py**
  - REST API with Lambda integration
  - Custom authorizers and API keys
  - Usage plans and throttling
  - Request validation and transformation
  - CORS configuration

## 🚀 Getting Started

### Prerequisites
- AWS CDK v2.x installed
- Node.js 18+ (for TypeScript examples)
- Python 3.8+ (for Python examples)
- AWS CLI configured with appropriate permissions

### TypeScript Setup
```bash
# Install dependencies
npm install

# Bootstrap CDK (if not done already)
cdk bootstrap

# Deploy an example stack
cdk deploy AdvancedAmplifyStack
```

### Python Setup
```bash
# Create virtual environment
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate

# Install dependencies
pip install aws-cdk-lib must-cdk

# Deploy an example stack
cdk deploy AdvancedAmplifyStack
```

## 📋 Example Details

### Advanced Amplify Configuration
Features demonstrated:
- **Multi-environment setup**: Production, staging, and development branches
- **Custom domains**: SSL certificates and subdomain routing
- **Build optimization**: Caching strategies and performance tuning
- **Security**: Environment variable management and access controls
- **CI/CD**: Automated testing and deployment pipelines

### ECS with CodeDeploy
Features demonstrated:
- **Blue/Green deployments**: Zero-downtime deployments with automatic rollback
- **Auto-scaling**: CPU and memory-based scaling policies
- **Load balancing**: Application Load Balancer with health checks
- **Monitoring**: CloudWatch integration and container insights
- **Security**: VPC configuration and security groups

### CloudFront Distribution
Features demonstrated:
- **Multi-origin setup**: Static assets from S3, API calls to API Gateway
- **Caching strategies**: Different cache behaviors for different content types
- **Security**: WAF integration, security headers, and geographic restrictions
- **Performance**: HTTP/2, compression, and edge location optimization
- **Monitoring**: Access logging and real-time metrics

### API Gateway with Lambda
Features demonstrated:
- **Authentication**: Custom authorizers with JWT token validation
- **Rate limiting**: API keys and usage plans with throttling
- **Validation**: Request/response validation and transformation
- **Error handling**: Custom error responses and CORS configuration
- **Integration**: DynamoDB integration with proper IAM permissions

## 🔧 Customization

Each example includes extensive configuration options that can be customized for your specific use case:

### Environment Variables
Update the environment-specific configurations in each example:
```typescript
// TypeScript
environmentVariables: {
  NODE_ENV: 'production',
  API_ENDPOINT: 'https://your-api.com',
  DATABASE_URL: 'your-database-connection'
}
```

```python
# Python
environment_variables={
    "NODE_ENV": "production",
    "API_ENDPOINT": "https://your-api.com",
    "DATABASE_URL": "your-database-connection"
}
```

### Domain Configuration
Replace placeholder domains with your actual domains:
```typescript
// TypeScript
customDomain: {
  domainName: 'your-domain.com',
  certificateArn: 'your-certificate-arn'
}
```

### Security Settings
Update security configurations for your requirements:
- WAF rules and IP restrictions
- API key management
- IAM roles and policies
- VPC and security group configurations

## 🏗️ Deployment Best Practices

### 1. Environment Separation
- Use different AWS accounts or regions for different environments
- Implement proper tagging strategies
- Use parameter stores for sensitive configuration

### 2. Security
- Enable encryption at rest and in transit
- Implement least privilege access policies
- Regular security audits and updates

### 3. Monitoring
- Set up CloudWatch alarms and dashboards
- Implement distributed tracing
- Configure log aggregation and analysis

### 4. Cost Optimization
- Use appropriate instance types and sizes
- Implement auto-scaling policies
- Regular cost reviews and optimization

## 📚 Additional Resources

- [AWS CDK Documentation](https://docs.aws.amazon.com/cdk/)
- [Must CDK API Reference](../docs/python/api.md)
- [AWS Well-Architected Framework](https://aws.amazon.com/architecture/well-architected/)
- [AWS Security Best Practices](https://aws.amazon.com/security/security-resources/)

## 🤝 Contributing

To contribute additional examples:
1. Follow the existing naming convention
2. Include both TypeScript and Python versions
3. Add comprehensive comments explaining the configuration
4. Update this README with example details
5. Test the example in a clean AWS environment

## 📞 Support

For questions or issues with these examples:
- Check the [Must CDK documentation](../README.md)
- Review AWS CDK troubleshooting guides
- Open an issue in the project repository
