# Amplify Examples

This directory contains examples for deploying AWS Amplify applications using Must CDK.

## Examples

### 1. Basic Amplify App
- **Files**: `basic-amplify.ts` / `basic_amplify.py`
- **Description**: Simple Amplify app deployment with automatic GitHub token detection
- **Features**:
  - Automatic GitHub CLI token detection
  - Basic build configuration
  - Environment variables setup

### 2. Advanced Amplify App
- **Files**: `amplify-advanced.ts` / `amplify_advanced.py`
- **Description**: Production-ready Amplify app with multiple environments
- **Features**:
  - Multi-environment branch configurations
  - Custom domain setup with SSL
  - Advanced build specifications for Next.js
  - Pull request previews
  - Environment-specific variables

### 3. Test Files
- **Files**: `test-amplify.ts` / `test-amplify.py`
- **Description**: Test scripts to validate your Amplify configuration without deploying
- **Usage**: 
  ```bash
  # TypeScript test
  cd examples/amplify
  npx ts-node test-amplify.ts
  
  # Python test (requires must-cdk package installed)
  cd examples/amplify
  python3 test-amplify.py
  ```

## GitHub Authentication

The Must CDK Amplify construct supports multiple authentication methods using GitHub personal access tokens.

### 1. GitHub CLI (Recommended for Development) 📱
```bash
# Install GitHub CLI
brew install gh  # macOS
# or
sudo apt install gh  # Ubuntu

# Authenticate
gh auth login

# Verify authentication
gh auth token
```

### 2. Environment Variable 🌍
```bash
export MUFIN_PUSH_TOKEN=your_github_personal_access_token
```

### 3. Direct Token (in code) 🔑
```typescript
// TypeScript
new AmplifyApp(this, 'MyApp', {
  appName: 'my-app',
  repository: 'https://github.com/user/repo',
  accessToken: 'your_github_token'
});
```

```python
# Python
AmplifyApp(
    self, "MyApp",
    app_name="my-app",
    repository="https://github.com/user/repo",
    access_token="your_github_token"
)
```

### 4. AWS Secrets Manager 🔒
```typescript
// TypeScript
import { SecretValue } from 'aws-cdk-lib';

new AmplifyApp(this, 'MyApp', {
  appName: 'my-app',
  repository: 'https://github.com/user/repo',
  accessToken: SecretValue.secretsManager('github-token').unsafeUnwrap()
});
```

## Production Migration Recommendation

💡 **After your project is successfully deployed and working**, consider migrating to GitHub Apps for enhanced security:

- **Better Security**: Fine-grained permissions and organization-level control
- **Higher Rate Limits**: Better API quotas for large organizations
- **Audit Trail**: Enhanced monitoring and logging capabilities
- **Organization Support**: Works better with GitHub Enterprise and organizations

For GitHub App setup guidance, see: https://docs.github.com/en/apps/creating-github-apps

## Testing Your Configuration

Before deploying, you can test your configuration using the provided test files:

### TypeScript Test
```bash
cd examples/amplify
npx ts-node test-amplify.ts
```

This will:
- Show the GitHub token detection process
- Generate the CloudFormation template
- Display any configuration issues

### Python Test
```bash
cd examples/amplify
python3 test-amplify.py
```

## Deployment

### TypeScript
```bash
npm install
cdk deploy BasicAmplifyStack
```

### Python
```bash
pip install aws-cdk-lib must-cdk
cdk deploy BasicAmplifyStack
```

## Troubleshooting

### "Invalid request provided: You should at least provide one valid token"
This error occurs when no GitHub token is available. Solutions:

1. **Check GitHub CLI authentication**:
   ```bash
   gh auth status
   gh auth token  # Should return a token
   ```

2. **Re-authenticate with GitHub CLI**:
   ```bash
   gh auth logout
   gh auth login
   ```

3. **Set environment variable**:
   ```bash
   export MUFIN_PUSH_TOKEN=ghp_your_token_here
   ```

4. **Create GitHub Personal Access Token**:
   - Go to GitHub Settings > Developer settings > Personal access tokens
   - Generate new token with `repo` scope
   - Use token in one of the methods above

### Build Failures
- Check your repository has the correct build commands
- Verify environment variables are set correctly
- Review Amplify build logs in AWS Console

### Domain Issues
- Ensure DNS is properly configured
- Verify SSL certificate is in the correct region (us-east-1 for CloudFront)
- Check domain ownership verification

## Debug Output

When you run the construct, you'll see detailed logging like:
```
🔍 Attempting to retrieve GitHub token...
📱 Trying to get token from GitHub CLI (gh auth token)...
✅ Successfully retrieved GitHub token from gh CLI
💡 Tip: For production deployments, consider migrating to GitHub Apps
   for better security and organization support after setup is complete.
🚀 Creating Amplify App with authenticated GitHub access...
```

This helps you understand exactly what's happening during the authentication process.
