# API Gateway Lambda Examples

This directory contains examples for creating REST APIs using Amazon API Gateway with AWS Lambda integration.

## Examples

### Comprehensive API Gateway with Lambda
- **Files**: `api-gateway-lambda.ts` / `api_gateway_lambda.py`
- **Description**: Production-ready REST API with authentication, rate limiting, and validation
- **Features**:
  - REST API with Lambda integration
  - Custom authorizers with token-based authentication
  - API keys and usage plans with rate limiting
  - Request validation and transformation
  - CORS configuration
  - DynamoDB integration
  - Comprehensive error handling

## Key Components

### 1. API Gateway Configuration
- REST API with regional endpoint
- Custom domain support
- CORS configuration
- Request/response validation

### 2. Lambda Functions
- Multiple functions for different operations
- Proper error handling
- Environment variable configuration
- DynamoDB integration with IAM permissions

### 3. Authentication & Authorization
- Custom Lambda authorizer
- Token-based authentication
- API key management
- Usage plans with throttling

### 4. Data Storage
- DynamoDB table with proper indexing
- IAM roles for Lambda access
- Point-in-time recovery enabled

## Deployment

```bash
# TypeScript
npm install
cdk deploy ApiGatewayLambdaStack

# Python
pip install aws-cdk-lib must-cdk
cdk deploy ApiGatewayLambdaStack
```

## API Usage

### Authentication
Include the authorization header:
```bash
curl -H "Authorization: Bearer valid-token" \
     -H "X-API-Key: your-api-key" \
     https://api.example.com/v1/users/123
```

### Available Endpoints
- `GET /users` - List all users
- `GET /users/{userId}` - Get specific user
- `POST /users/{userId}` - Create/update user

## Customization

### Lambda Functions
Update function code and configuration:
```typescript
const myFunction = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_18_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset('lambda'),
  environment: {
    TABLE_NAME: table.tableName
  }
});
```

### API Resources
Add new resources and methods:
```typescript
resources: [
  {
    pathPart: 'products',
    methods: [
      {
        httpMethod: 'GET',
        integration: {
          type: 'AWS_PROXY',
          lambdaFunction: getProductsFunction
        },
        authorization: {
          authorizationType: 'CUSTOM',
          authorizerId: 'TokenAuthorizer'
        }
      }
    ]
  }
]
```

### Usage Plans
Configure rate limiting:
```typescript
usagePlans: [
  {
    planName: 'premium-plan',
    throttle: {
      rateLimit: 1000,
      burstLimit: 2000
    },
    quota: {
      limit: 100000,
      period: 'MONTH'
    }
  }
]
```

## Security Best Practices

### Authentication
- Use strong token validation
- Implement token expiration
- Regular key rotation
- Audit access logs

### Authorization
- Implement fine-grained permissions
- Use least privilege principle
- Regular permission reviews
- Monitor unauthorized access attempts

### Data Protection
- Encrypt data at rest and in transit
- Input validation and sanitization
- SQL injection prevention
- Rate limiting and DDoS protection

## Monitoring and Debugging

### CloudWatch Metrics
- API Gateway metrics (latency, errors, throttles)
- Lambda metrics (duration, errors, throttles)
- DynamoDB metrics (read/write capacity, throttles)

### Logging
- API Gateway access logs
- Lambda function logs
- Custom application logs
- Structured logging for better analysis

### Debugging
- Enable X-Ray tracing
- Use CloudWatch Insights
- Monitor error rates and patterns
- Set up alerts for critical metrics

## Testing

### Unit Testing
```bash
# Test Lambda functions locally
npm test

# Test with SAM CLI
sam local start-api
```

### Integration Testing
```bash
# Test API endpoints
curl -X GET https://api.example.com/v1/users \
     -H "Authorization: Bearer test-token" \
     -H "X-API-Key: test-key"
```

### Load Testing
- Use tools like Artillery or JMeter
- Test rate limiting behavior
- Monitor performance under load
- Validate auto-scaling behavior
