import * as cdk from 'aws-cdk-lib';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as dynamodb from 'aws-cdk-lib/aws-dynamodb';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { ApiGatewayLambda } from 'must-cdk';

export class ApiGatewayLambdaStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Create DynamoDB table for data storage
    const dataTable = new dynamodb.Table(this, 'DataTable', {
      tableName: 'user-data',
      partitionKey: {
        name: 'userId',
        type: dynamodb.AttributeType.STRING
      },
      sortKey: {
        name: 'timestamp',
        type: dynamodb.AttributeType.NUMBER
      },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
      encryption: dynamodb.TableEncryption.AWS_MANAGED,
      pointInTimeRecovery: true,
      removalPolicy: cdk.RemovalPolicy.DESTROY // Use RETAIN for production
    });

    // Create Lambda functions for different API operations
    const getUserFunction = new lambda.Function(this, 'GetUserFunction', {
      runtime: lambda.Runtime.NODEJS_18_X,
      handler: 'index.handler',
      code: lambda.Code.fromInline(`
        const AWS = require('aws-sdk');
        const dynamodb = new AWS.DynamoDB.DocumentClient();

        exports.handler = async (event) => {
          const userId = event.pathParameters.userId;
          
          try {
            const result = await dynamodb.query({
              TableName: process.env.TABLE_NAME,
              KeyConditionExpression: 'userId = :userId',
              ExpressionAttributeValues: {
                ':userId': userId
              }
            }).promise();

            return {
              statusCode: 200,
              headers: {
                'Content-Type': 'application/json',
                'Access-Control-Allow-Origin': '*'
              },
              body: JSON.stringify(result.Items)
            };
          } catch (error) {
            return {
              statusCode: 500,
              body: JSON.stringify({ error: error.message })
            };
          }
        };
      `),
      environment: {
        TABLE_NAME: dataTable.tableName
      },
      timeout: cdk.Duration.seconds(30)
    });

    const createUserFunction = new lambda.Function(this, 'CreateUserFunction', {
      runtime: lambda.Runtime.NODEJS_18_X,
      handler: 'index.handler',
      code: lambda.Code.fromInline(`
        const AWS = require('aws-sdk');
        const dynamodb = new AWS.DynamoDB.DocumentClient();

        exports.handler = async (event) => {
          const data = JSON.parse(event.body);
          const userId = event.pathParameters.userId;
          
          try {
            await dynamodb.put({
              TableName: process.env.TABLE_NAME,
              Item: {
                userId: userId,
                timestamp: Date.now(),
                ...data
              }
            }).promise();

            return {
              statusCode: 201,
              headers: {
                'Content-Type': 'application/json',
                'Access-Control-Allow-Origin': '*'
              },
              body: JSON.stringify({ message: 'User created successfully' })
            };
          } catch (error) {
            return {
              statusCode: 500,
              body: JSON.stringify({ error: error.message })
            };
          }
        };
      `),
      environment: {
        TABLE_NAME: dataTable.tableName
      },
      timeout: cdk.Duration.seconds(30)
    });

    // Grant DynamoDB permissions to Lambda functions
    dataTable.grantReadData(getUserFunction);
    dataTable.grantWriteData(createUserFunction);

    // Create custom authorizer Lambda
    const authorizerFunction = new lambda.Function(this, 'AuthorizerFunction', {
      runtime: lambda.Runtime.NODEJS_18_X,
      handler: 'index.handler',
      code: lambda.Code.fromInline(`
        exports.handler = async (event) => {
          const token = event.authorizationToken;
          
          // Simple token validation (replace with your auth logic)
          const isValid = token === 'Bearer valid-token';
          
          return {
            principalId: 'user123',
            policyDocument: {
              Version: '2012-10-17',
              Statement: [
                {
                  Action: 'execute-api:Invoke',
                  Effect: isValid ? 'Allow' : 'Deny',
                  Resource: event.methodArn
                }
              ]
            },
            context: {
              userId: 'user123',
              role: 'admin'
            }
          };
        };
      `),
      timeout: cdk.Duration.seconds(10)
    });

    // Create comprehensive API Gateway with Lambda integration
    const api = new ApiGatewayLambda(this, 'UserApi', {
      apiName: 'user-management-api',
      description: 'Comprehensive API for user management with authentication',
      
      // API configuration
      apiType: 'REST',
      endpointType: 'REGIONAL',
      
      // Custom domain configuration
      customDomain: {
        domainName: 'api.example.com',
        certificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
        basePath: 'v1'
      },

      // CORS configuration
      corsOptions: {
        allowOrigins: ['https://app.example.com', 'https://admin.example.com'],
        allowMethods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
        allowHeaders: ['Content-Type', 'Authorization', 'X-Api-Key'],
        allowCredentials: true,
        maxAge: cdk.Duration.hours(1)
      },

      // Authentication and authorization
      authorizers: [
        {
          authorizerName: 'TokenAuthorizer',
          authorizerFunction: authorizerFunction,
          authorizerType: 'TOKEN',
          identitySource: 'method.request.header.Authorization',
          authorizerResultTtlInSeconds: 300
        }
      ],

      // API Key and usage plans
      apiKeys: [
        {
          apiKeyName: 'admin-key',
          description: 'API key for admin access'
        },
        {
          apiKeyName: 'client-key',
          description: 'API key for client applications'
        }
      ],

      usagePlans: [
        {
          planName: 'admin-plan',
          description: 'Usage plan for admin users',
          throttle: {
            rateLimit: 1000,
            burstLimit: 2000
          },
          quota: {
            limit: 100000,
            period: 'MONTH'
          },
          apiKeys: ['admin-key']
        },
        {
          planName: 'client-plan',
          description: 'Usage plan for client applications',
          throttle: {
            rateLimit: 100,
            burstLimit: 200
          },
          quota: {
            limit: 10000,
            period: 'MONTH'
          },
          apiKeys: ['client-key']
        }
      ],

      // Resource and method definitions
      resources: [
        {
          pathPart: 'users',
          methods: [
            {
              httpMethod: 'GET',
              integration: {
                type: 'AWS_PROXY',
                lambdaFunction: getUserFunction
              },
              authorization: {
                authorizationType: 'CUSTOM',
                authorizerId: 'TokenAuthorizer'
              },
              apiKeyRequired: true,
              requestParameters: {
                'method.request.querystring.limit': false,
                'method.request.querystring.offset': false
              },
              requestValidation: {
                validateRequestBody: false,
                validateRequestParameters: true
              }
            }
          ],
          childResources: [
            {
              pathPart: '{userId}',
              methods: [
                {
                  httpMethod: 'GET',
                  integration: {
                    type: 'AWS_PROXY',
                    lambdaFunction: getUserFunction
                  },
                  authorization: {
                    authorizationType: 'CUSTOM',
                    authorizerId: 'TokenAuthorizer'
                  },
                  apiKeyRequired: true,
                  requestParameters: {
                    'method.request.path.userId': true
                  }
                },
                {
                  httpMethod: 'POST',
                  integration: {
                    type: 'AWS_PROXY',
                    lambdaFunction: createUserFunction
                  },
                  authorization: {
                    authorizationType: 'CUSTOM',
                    authorizerId: 'TokenAuthorizer'
                  },
                  apiKeyRequired: true,
                  requestParameters: {
                    'method.request.path.userId': true
                  },
                  requestValidation: {
                    validateRequestBody: true,
                    requestBodySchema: {
                      type: 'object',
                      required: ['name', 'email'],
                      properties: {
                        name: { type: 'string' },
                        email: { type: 'string', format: 'email' },
                        age: { type: 'number', minimum: 0 }
                      }
                    }
                  }
                }
              ]
            }
          ]
        }
      ],

      // Monitoring and logging
      enableAccessLogging: true,
      enableExecutionLogging: true,
      logLevel: 'INFO',
      dataTraceEnabled: true,
      metricsEnabled: true,

      // Request/Response transformations
      requestTemplates: {
        'application/json': `{
          "userId": "$input.params('userId')",
          "body": $input.json('$'),
          "headers": {
            #foreach($header in $input.params().header.keySet())
            "$header": "$util.escapeJavaScript($input.params().header.get($header))"
            #if($foreach.hasNext),#end
            #end
          }
        }`
      },

      // Error handling
      gatewayResponses: [
        {
          type: 'UNAUTHORIZED',
          statusCode: '401',
          responseTemplates: {
            'application/json': '{"message": "Unauthorized access"}'
          },
          responseHeaders: {
            'Access-Control-Allow-Origin': "'*'"
          }
        },
        {
          type: 'ACCESS_DENIED',
          statusCode: '403',
          responseTemplates: {
            'application/json': '{"message": "Access denied"}'
          }
        }
      ]
    });

    // Output important values
    new cdk.CfnOutput(this, 'ApiUrl', {
      value: api.restApi.url,
      description: 'API Gateway URL'
    });

    new cdk.CfnOutput(this, 'ApiId', {
      value: api.restApi.restApiId,
      description: 'API Gateway ID'
    });

    new cdk.CfnOutput(this, 'TableName', {
      value: dataTable.tableName,
      description: 'DynamoDB Table Name'
    });
  }
}
