from aws_cdk import (
    App,
    Stack,
    StackProps,
    CfnOutput,
    Duration,
    RemovalPolicy
)
from aws_cdk import aws_lambda as lambda_
from aws_cdk import aws_dynamodb as dynamodb
from aws_cdk import aws_iam as iam
from constructs import Construct
from must_cdk import ApiGatewayLambda

class ApiGatewayLambdaStack(Stack):
    def __init__(self, scope: Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        # Create DynamoDB table for data storage
        data_table = dynamodb.Table(
            self, "DataTable",
            table_name="user-data",
            partition_key=dynamodb.Attribute(
                name="userId",
                type=dynamodb.AttributeType.STRING
            ),
            sort_key=dynamodb.Attribute(
                name="timestamp",
                type=dynamodb.AttributeType.NUMBER
            ),
            billing_mode=dynamodb.BillingMode.PAY_PER_REQUEST,
            encryption=dynamodb.TableEncryption.AWS_MANAGED,
            point_in_time_recovery=True,
            removal_policy=RemovalPolicy.DESTROY  # Use RETAIN for production
        )

        # Create Lambda functions for different API operations
        get_user_function = lambda_.Function(
            self, "GetUserFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("""
                const AWS = require('aws-sdk');
                const dynamodb = new AWS.DynamoDB.DocumentClient();

                exports.handler = async (event) => {
                  const userId = event.pathParameters.userId;
                  
                  try {
                    const result = await dynamodb.query({
                      TableName: process.env.TABLE_NAME,
                      KeyConditionExpression: 'userId = :userId',
                      ExpressionAttributeValues: {
                        ':userId': userId
                      }
                    }).promise();

                    return {
                      statusCode: 200,
                      headers: {
                        'Content-Type': 'application/json',
                        'Access-Control-Allow-Origin': '*'
                      },
                      body: JSON.stringify(result.Items)
                    };
                  } catch (error) {
                    return {
                      statusCode: 500,
                      body: JSON.stringify({ error: error.message })
                    };
                  }
                };
            """),
            environment={
                "TABLE_NAME": data_table.table_name
            },
            timeout=Duration.seconds(30)
        )

        create_user_function = lambda_.Function(
            self, "CreateUserFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("""
                const AWS = require('aws-sdk');
                const dynamodb = new AWS.DynamoDB.DocumentClient();

                exports.handler = async (event) => {
                  const data = JSON.parse(event.body);
                  const userId = event.pathParameters.userId;
                  
                  try {
                    await dynamodb.put({
                      TableName: process.env.TABLE_NAME,
                      Item: {
                        userId: userId,
                        timestamp: Date.now(),
                        ...data
                      }
                    }).promise();

                    return {
                      statusCode: 201,
                      headers: {
                        'Content-Type': 'application/json',
                        'Access-Control-Allow-Origin': '*'
                      },
                      body: JSON.stringify({ message: 'User created successfully' })
                    };
                  } catch (error) {
                    return {
                      statusCode: 500,
                      body: JSON.stringify({ error: error.message })
                    };
                  }
                };
            """),
            environment={
                "TABLE_NAME": data_table.table_name
            },
            timeout=Duration.seconds(30)
        )

        # Grant DynamoDB permissions to Lambda functions
        data_table.grant_read_data(get_user_function)
        data_table.grant_write_data(create_user_function)

        # Create custom authorizer Lambda
        authorizer_function = lambda_.Function(
            self, "AuthorizerFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("""
                exports.handler = async (event) => {
                  const token = event.authorizationToken;
                  
                  // Simple token validation (replace with your auth logic)
                  const isValid = token === 'Bearer valid-token';
                  
                  return {
                    principalId: 'user123',
                    policyDocument: {
                      Version: '2012-10-17',
                      Statement: [
                        {
                          Action: 'execute-api:Invoke',
                          Effect: isValid ? 'Allow' : 'Deny',
                          Resource: event.methodArn
                        }
                      ]
                    },
                    context: {
                      userId: 'user123',
                      role: 'admin'
                    }
                  };
                };
            """),
            timeout=Duration.seconds(10)
        )

        # Create comprehensive API Gateway with Lambda integration
        api = ApiGatewayLambda(
            self, "UserApi",
            api_name="user-management-api",
            description="Comprehensive API for user management with authentication",
            
            # API configuration
            api_type="REST",
            endpoint_type="REGIONAL",
            
            # Custom domain configuration
            custom_domain={
                "domain_name": "api.example.com",
                "certificate_arn": "arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012",
                "base_path": "v1"
            },

            # CORS configuration
            cors_options={
                "allow_origins": ["https://app.example.com", "https://admin.example.com"],
                "allow_methods": ["GET", "POST", "PUT", "DELETE", "OPTIONS"],
                "allow_headers": ["Content-Type", "Authorization", "X-Api-Key"],
                "allow_credentials": True,
                "max_age": Duration.hours(1)
            },

            # Authentication and authorization
            authorizers=[
                {
                    "authorizer_name": "TokenAuthorizer",
                    "authorizer_function": authorizer_function,
                    "authorizer_type": "TOKEN",
                    "identity_source": "method.request.header.Authorization",
                    "authorizer_result_ttl_in_seconds": 300
                }
            ],

            # API Key and usage plans
            api_keys=[
                {
                    "api_key_name": "admin-key",
                    "description": "API key for admin access"
                },
                {
                    "api_key_name": "client-key",
                    "description": "API key for client applications"
                }
            ],

            usage_plans=[
                {
                    "plan_name": "admin-plan",
                    "description": "Usage plan for admin users",
                    "throttle": {
                        "rate_limit": 1000,
                        "burst_limit": 2000
                    },
                    "quota": {
                        "limit": 100000,
                        "period": "MONTH"
                    },
                    "api_keys": ["admin-key"]
                },
                {
                    "plan_name": "client-plan",
                    "description": "Usage plan for client applications",
                    "throttle": {
                        "rate_limit": 100,
                        "burst_limit": 200
                    },
                    "quota": {
                        "limit": 10000,
                        "period": "MONTH"
                    },
                    "api_keys": ["client-key"]
                }
            ],

            # Resource and method definitions
            resources=[
                {
                    "path_part": "users",
                    "methods": [
                        {
                            "http_method": "GET",
                            "integration": {
                                "type": "AWS_PROXY",
                                "lambda_function": get_user_function
                            },
                            "authorization": {
                                "authorization_type": "CUSTOM",
                                "authorizer_id": "TokenAuthorizer"
                            },
                            "api_key_required": True,
                            "request_parameters": {
                                "method.request.querystring.limit": False,
                                "method.request.querystring.offset": False
                            },
                            "request_validation": {
                                "validate_request_body": False,
                                "validate_request_parameters": True
                            }
                        }
                    ],
                    "child_resources": [
                        {
                            "path_part": "{userId}",
                            "methods": [
                                {
                                    "http_method": "GET",
                                    "integration": {
                                        "type": "AWS_PROXY",
                                        "lambda_function": get_user_function
                                    },
                                    "authorization": {
                                        "authorization_type": "CUSTOM",
                                        "authorizer_id": "TokenAuthorizer"
                                    },
                                    "api_key_required": True,
                                    "request_parameters": {
                                        "method.request.path.userId": True
                                    }
                                },
                                {
                                    "http_method": "POST",
                                    "integration": {
                                        "type": "AWS_PROXY",
                                        "lambda_function": create_user_function
                                    },
                                    "authorization": {
                                        "authorization_type": "CUSTOM",
                                        "authorizer_id": "TokenAuthorizer"
                                    },
                                    "api_key_required": True,
                                    "request_parameters": {
                                        "method.request.path.userId": True
                                    },
                                    "request_validation": {
                                        "validate_request_body": True,
                                        "request_body_schema": {
                                            "type": "object",
                                            "required": ["name", "email"],
                                            "properties": {
                                                "name": {"type": "string"},
                                                "email": {"type": "string", "format": "email"},
                                                "age": {"type": "number", "minimum": 0}
                                            }
                                        }
                                    }
                                }
                            ]
                        }
                    ]
                }
            ],

            # Monitoring and logging
            enable_access_logging=True,
            enable_execution_logging=True,
            log_level="INFO",
            data_trace_enabled=True,
            metrics_enabled=True,

            # Request/Response transformations
            request_templates={
                "application/json": """{
                  "userId": "$input.params('userId')",
                  "body": $input.json('$'),
                  "headers": {
                    #foreach($header in $input.params().header.keySet())
                    "$header": "$util.escapeJavaScript($input.params().header.get($header))"
                    #if($foreach.hasNext),#end
                    #end
                  }
                }"""
            },

            # Error handling
            gateway_responses=[
                {
                    "type": "UNAUTHORIZED",
                    "status_code": "401",
                    "response_templates": {
                        "application/json": '{"message": "Unauthorized access"}'
                    },
                    "response_headers": {
                        "Access-Control-Allow-Origin": "'*'"
                    }
                },
                {
                    "type": "ACCESS_DENIED",
                    "status_code": "403",
                    "response_templates": {
                        "application/json": '{"message": "Access denied"}'
                    }
                }
            ]
        )

        # Output important values
        CfnOutput(
            self, "ApiUrl",
            value=api.rest_api.url,
            description="API Gateway URL"
        )

        CfnOutput(
            self, "ApiId",
            value=api.rest_api.rest_api_id,
            description="API Gateway ID"
        )

        CfnOutput(
            self, "TableName",
            value=data_table.table_name,
            description="DynamoDB Table Name"
        )

# Example usage
app = App()
ApiGatewayLambdaStack(app, "ApiGatewayLambdaStack")
app.synth()
