from aws_cdk import (
    App,
    Stack,
    StackProps,
    CfnOutput,
    Duration,
    RemovalPolicy
)
from aws_cdk import aws_s3 as s3
from aws_cdk import aws_apigateway as apigateway
from aws_cdk import aws_lambda as lambda_
from constructs import Construct
from must_cdk import CloudFrontDistribution

class CloudFrontDistributionStack(Stack):
    def __init__(self, scope: Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        # Create S3 bucket for static assets
        static_assets_bucket = s3.Bucket(
            self, "StaticAssetsBucket",
            bucket_name="my-app-static-assets",
            versioned=True,
            encryption=s3.BucketEncryption.S3_MANAGED,
            block_public_access=s3.BlockPublicAccess.BLOCK_ALL,
            removal_policy=RemovalPolicy.DESTROY  # Use RETAIN for production
        )

        # Create Lambda function for API
        api_lambda = lambda_.Function(
            self, "ApiLambda",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("""
                exports.handler = async (event) => {
                  return {
                    statusCode: 200,
                    headers: {
                      'Content-Type': 'application/json',
                      'Access-Control-Allow-Origin': '*'
                    },
                    body: JSON.stringify({
                      message: 'Hello from API!',
                      timestamp: new Date().toISOString()
                    })
                  };
                };
            """),
            timeout=Duration.seconds(30)
        )

        # Create API Gateway
        api = apigateway.RestApi(
            self, "Api",
            rest_api_name="My App API",
            description="API for my application",
            default_cors_preflight_options=apigateway.CorsOptions(
                allow_origins=apigateway.Cors.ALL_ORIGINS,
                allow_methods=apigateway.Cors.ALL_METHODS,
                allow_headers=["Content-Type", "Authorization"]
            )
        )

        api_integration = apigateway.LambdaIntegration(api_lambda)
        api.root.add_resource("api").add_method("GET", api_integration)

        # Create CloudFront distribution with multiple origins
        distribution = CloudFrontDistribution(
            self, "AppDistribution",
            comment="Multi-origin CloudFront distribution for web app",
            
            # Primary origin (S3 for static assets)
            default_origin={
                "domain_name": static_assets_bucket.bucket_domain_name,
                "origin_path": "/static",
                "custom_origin_config": {
                    "http_port": 80,
                    "https_port": 443,
                    "origin_protocol_policy": "https-only"
                }
            },

            # Additional origins
            additional_origins=[
                {
                    "id": "api-origin",
                    "domain_name": f"{api.rest_api_id}.execute-api.{self.region}.amazonaws.com",
                    "origin_path": f"/{api.deployment_stage.stage_name}",
                    "custom_origin_config": {
                        "http_port": 80,
                        "https_port": 443,
                        "origin_protocol_policy": "https-only",
                        "origin_ssl_protocols": ["TLSv1.2"]
                    }
                }
            ],

            # Cache behaviors
            default_cache_behavior={
                "target_origin_id": "default",
                "viewer_protocol_policy": "redirect-to-https",
                "allowed_methods": ["GET", "HEAD", "OPTIONS"],
                "cached_methods": ["GET", "HEAD"],
                "compress": True,
                "cache_policy_id": "4135ea2d-6df8-44a3-9df3-4b5a84be39ad",  # CachingOptimized
                "origin_request_policy_id": "88a5eaf4-2fd4-4709-b370-b4c650ea3fcf",  # CORS-S3Origin
                "response_headers_policy_id": "5cc3b908-e619-4b99-88e5-2cf7f45965bd"  # SecurityHeadersPolicy
            },

            additional_cache_behaviors=[
                {
                    "path_pattern": "/api/*",
                    "target_origin_id": "api-origin",
                    "viewer_protocol_policy": "redirect-to-https",
                    "allowed_methods": ["GET", "HEAD", "OPTIONS", "PUT", "POST", "PATCH", "DELETE"],
                    "cached_methods": ["GET", "HEAD"],
                    "compress": True,
                    "cache_policy_id": "4135ea2d-6df8-44a3-9df3-4b5a84be39ad",
                    "origin_request_policy_id": "b689b0a8-53d0-40ab-baf2-68738e2966ac",  # AllViewer
                    "response_headers_policy_id": "5cc3b908-e619-4b99-88e5-2cf7f45965bd"
                }
            ],

            # Custom domain configuration
            custom_domain={
                "domain_name": "app.example.com",
                "certificate_arn": "arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012",
                "aliases": ["www.app.example.com"]
            },

            # Security configuration
            web_acl_id="arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a",
            
            # Geographic restrictions
            geo_restriction={
                "restriction_type": "whitelist",
                "locations": ["US", "CA", "GB", "DE", "FR"]
            },

            # Error pages
            custom_error_responses=[
                {
                    "error_code": 404,
                    "response_code": 200,
                    "response_page_path": "/index.html",
                    "error_caching_min_ttl": 300
                },
                {
                    "error_code": 403,
                    "response_code": 200,
                    "response_page_path": "/index.html",
                    "error_caching_min_ttl": 300
                }
            ],

            # Logging configuration
            enable_logging=True,
            log_bucket=static_assets_bucket,
            log_file_prefix="cloudfront-logs/",
            log_include_cookies=False,

            # Performance settings
            price_class="PriceClass_100",  # Use only North America and Europe edge locations
            http_version="http2",
            enable_ipv6=True,

            # Additional security headers
            response_headers_policy={
                "security_headers_config": {
                    "strict_transport_security": {
                        "access_control_max_age_sec": 31536000,
                        "include_subdomains": True,
                        "override": True
                    },
                    "content_type_options": {
                        "override": True
                    },
                    "frame_options": {
                        "frame_option": "DENY",
                        "override": True
                    },
                    "referrer_policy": {
                        "referrer_policy": "strict-origin-when-cross-origin",
                        "override": True
                    }
                }
            }
        )

        # Grant CloudFront access to S3 bucket
        static_assets_bucket.grant_read(distribution.origin_access_identity)

        # Output important values
        CfnOutput(
            self, "DistributionDomainName",
            value=distribution.distribution_domain_name,
            description="CloudFront Distribution Domain Name"
        )

        CfnOutput(
            self, "DistributionId",
            value=distribution.distribution_id,
            description="CloudFront Distribution ID"
        )

        CfnOutput(
            self, "S3BucketName",
            value=static_assets_bucket.bucket_name,
            description="S3 Bucket Name for Static Assets"
        )

# Example usage
app = App()
CloudFrontDistributionStack(app, "CloudFrontDistributionStack")
app.synth()
