import * as cdk from 'aws-cdk-lib';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import { Construct } from 'constructs';
import { EcsCodeDeployService } from 'must-cdk';

export class EcsCodeDeployStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Create VPC for ECS cluster
    const vpc = new ec2.Vpc(this, 'EcsVpc', {
      maxAzs: 3,
      natGateways: 2,
      subnetConfiguration: [
        {
          cidrMask: 24,
          name: 'Public',
          subnetType: ec2.SubnetType.PUBLIC,
        },
        {
          cidrMask: 24,
          name: 'Private',
          subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        }
      ]
    });

    // Create ECS cluster
    const cluster = new ecs.Cluster(this, 'EcsCluster', {
      vpc,
      clusterName: 'production-cluster',
      containerInsights: true
    });

    // ECS service with Blue/Green deployment using CodeDeploy
    const ecsService = new EcsCodeDeployService(this, 'WebApiService', {
      cluster,
      serviceName: 'web-api-service',
      
      // Task definition configuration
      taskDefinition: {
        family: 'web-api-task',
        cpu: 1024,
        memoryLimitMiB: 2048,
        containers: [
          {
            name: 'web-api',
            image: 'nginx:latest', // Replace with your application image
            portMappings: [
              {
                containerPort: 80,
                protocol: ecs.Protocol.TCP
              }
            ],
            environment: {
              NODE_ENV: 'production',
              PORT: '80'
            },
            logging: ecs.LogDrivers.awsLogs({
              streamPrefix: 'web-api',
              logRetention: 7
            }),
            healthCheck: {
              command: ['CMD-SHELL', 'curl -f http://localhost/health || exit 1'],
              interval: cdk.Duration.seconds(30),
              timeout: cdk.Duration.seconds(5),
              retries: 3,
              startPeriod: cdk.Duration.seconds(60)
            }
          }
        ]
      },

      // Service configuration
      desiredCount: 3,
      minHealthyPercent: 50,
      maxHealthyPercent: 200,
      
      // Load balancer configuration
      loadBalancer: {
        internetFacing: true,
        healthCheckPath: '/health',
        healthCheckIntervalSeconds: 30,
        healthyThresholdCount: 2,
        unhealthyThresholdCount: 5,
        targetGroupPort: 80
      },

      // Auto scaling configuration
      autoScaling: {
        minCapacity: 2,
        maxCapacity: 10,
        targetCpuUtilization: 70,
        targetMemoryUtilization: 80,
        scaleInCooldown: cdk.Duration.minutes(5),
        scaleOutCooldown: cdk.Duration.minutes(2)
      },

      // CodeDeploy Blue/Green deployment configuration
      deploymentConfig: {
        deploymentConfigName: 'CodeDeployDefault.ECSAllAtOnceBlueGreen',
        terminationWaitTimeInMinutes: 5,
        deploymentReadyWaitTimeInMinutes: 0,
        blueGreenDeploymentConfiguration: {
          terminateBlueInstancesOnDeploymentSuccess: {
            action: 'TERMINATE',
            terminationWaitTimeInMinutes: 5
          },
          deploymentReadyOption: {
            actionOnTimeout: 'CONTINUE_DEPLOYMENT'
          },
          greenFleetProvisioningOption: {
            action: 'COPY_AUTO_SCALING_GROUP'
          }
        }
      },

      // Security configuration
      enableExecuteCommand: false,
      enableLogging: true,
      
      // Network configuration
      assignPublicIp: false,
      securityGroups: [
        new ec2.SecurityGroup(this, 'EcsSecurityGroup', {
          vpc,
          description: 'Security group for ECS service',
          allowAllOutbound: true
        })
      ]
    });

    // Configure security group rules
    ecsService.securityGroup.addIngressRule(
      ec2.Peer.anyIpv4(),
      ec2.Port.tcp(80),
      'Allow HTTP traffic from ALB'
    );

    // Output important values
    new cdk.CfnOutput(this, 'LoadBalancerDnsName', {
      value: ecsService.loadBalancer.loadBalancerDnsName,
      description: 'Load Balancer DNS Name'
    });

    new cdk.CfnOutput(this, 'ServiceArn', {
      value: ecsService.service.serviceArn,
      description: 'ECS Service ARN'
    });
  }
}
