from aws_cdk import (
    App,
    Stack,
    StackProps,
    CfnOutput,
    Duration
)
from aws_cdk import aws_ec2 as ec2
from aws_cdk import aws_ecs as ecs
from constructs import Construct
from must_cdk import EcsCodeDeployService

class EcsCodeDeployStack(Stack):
    def __init__(self, scope: Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        # Create VPC for ECS cluster
        vpc = ec2.Vpc(
            self, "EcsVpc",
            max_azs=3,
            nat_gateways=2,
            subnet_configuration=[
                ec2.SubnetConfiguration(
                    cidr_mask=24,
                    name="Public",
                    subnet_type=ec2.SubnetType.PUBLIC
                ),
                ec2.SubnetConfiguration(
                    cidr_mask=24,
                    name="Private",
                    subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
                )
            ]
        )

        # Create ECS cluster
        cluster = ecs.Cluster(
            self, "EcsCluster",
            vpc=vpc,
            cluster_name="production-cluster",
            container_insights=True
        )

        # Create security group for ECS service
        security_group = ec2.SecurityGroup(
            self, "EcsSecurityGroup",
            vpc=vpc,
            description="Security group for ECS service",
            allow_all_outbound=True
        )

        # ECS service with Blue/Green deployment using CodeDeploy
        ecs_service = EcsCodeDeployService(
            self, "WebApiService",
            cluster=cluster,
            service_name="web-api-service",
            
            # Task definition configuration
            task_definition={
                "family": "web-api-task",
                "cpu": 1024,
                "memory_limit_mib": 2048,
                "containers": [
                    {
                        "name": "web-api",
                        "image": "nginx:latest",  # Replace with your application image
                        "port_mappings": [
                            {
                                "container_port": 80,
                                "protocol": ecs.Protocol.TCP
                            }
                        ],
                        "environment": {
                            "NODE_ENV": "production",
                            "PORT": "80"
                        },
                        "logging": ecs.LogDrivers.aws_logs(
                            stream_prefix="web-api",
                            log_retention=7
                        ),
                        "health_check": {
                            "command": ["CMD-SHELL", "curl -f http://localhost/health || exit 1"],
                            "interval": Duration.seconds(30),
                            "timeout": Duration.seconds(5),
                            "retries": 3,
                            "start_period": Duration.seconds(60)
                        }
                    }
                ]
            },

            # Service configuration
            desired_count=3,
            min_healthy_percent=50,
            max_healthy_percent=200,
            
            # Load balancer configuration
            load_balancer={
                "internet_facing": True,
                "health_check_path": "/health",
                "health_check_interval_seconds": 30,
                "healthy_threshold_count": 2,
                "unhealthy_threshold_count": 5,
                "target_group_port": 80
            },

            # Auto scaling configuration
            auto_scaling={
                "min_capacity": 2,
                "max_capacity": 10,
                "target_cpu_utilization": 70,
                "target_memory_utilization": 80,
                "scale_in_cooldown": Duration.minutes(5),
                "scale_out_cooldown": Duration.minutes(2)
            },

            # CodeDeploy Blue/Green deployment configuration
            deployment_config={
                "deployment_config_name": "CodeDeployDefault.ECSAllAtOnceBlueGreen",
                "termination_wait_time_in_minutes": 5,
                "deployment_ready_wait_time_in_minutes": 0,
                "blue_green_deployment_configuration": {
                    "terminate_blue_instances_on_deployment_success": {
                        "action": "TERMINATE",
                        "termination_wait_time_in_minutes": 5
                    },
                    "deployment_ready_option": {
                        "action_on_timeout": "CONTINUE_DEPLOYMENT"
                    },
                    "green_fleet_provisioning_option": {
                        "action": "COPY_AUTO_SCALING_GROUP"
                    }
                }
            },

            # Security configuration
            enable_execute_command=False,
            enable_logging=True,
            
            # Network configuration
            assign_public_ip=False,
            security_groups=[security_group]
        )

        # Configure security group rules
        security_group.add_ingress_rule(
            ec2.Peer.any_ipv4(),
            ec2.Port.tcp(80),
            "Allow HTTP traffic from ALB"
        )

        # Output important values
        CfnOutput(
            self, "LoadBalancerDnsName",
            value=ecs_service.load_balancer.load_balancer_dns_name,
            description="Load Balancer DNS Name"
        )

        CfnOutput(
            self, "ServiceArn",
            value=ecs_service.service.service_arn,
            description="ECS Service ARN"
        )

# Example usage
app = App()
EcsCodeDeployStack(app, "EcsCodeDeployStack")
app.synth()
