"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmplifyApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const amplify = require("aws-cdk-lib/aws-amplify");
const constructs_1 = require("constructs");
class AmplifyApp extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        // Get GitHub authentication
        const accessToken = this.getGitHubAuthentication(props);
        if (!accessToken) {
            throw new Error(`
❌ GitHub authentication is required for Amplify App deployment!

Please provide authentication using one of these methods:

1. 🔑 Personal Access Token (Current approach):
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: 'your-github-token'
   })

2. 📱 Use GitHub CLI (for development):
   gh auth login
   
3. 🌍 Set environment variable:
   export MUFIN_PUSH_TOKEN=your-github-token

4. 🔒 Use AWS Secrets Manager:
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: SecretValue.secretsManager('github-token').unsafeUnwrap()
   })

💡 Production Recommendation:
   After your project is set up and working, consider migrating to GitHub Apps
   for better security, organization support, and higher rate limits.
   See: https://docs.github.com/en/apps/creating-github-apps
      `);
        }
        console.log("🚀 Creating Amplify App with authenticated GitHub access...");
        // Create Amplify App
        this.app = new amplify.CfnApp(this, "App", {
            name: props.appName,
            repository: props.repository,
            accessToken: accessToken,
            environmentVariables: this.formatEnvVars(props.environmentVariables),
            buildSpec: props.buildSpec || this.getDefaultBuildSpec(props.platform),
            platform: props.platform || "WEB_COMPUTE",
            customRules: props.customRules || this.getDefaultCustomRules(),
            autoBranchCreationConfig: {
                enableAutoBuild: props.buildSettings?.enableBranchAutoBuild ?? false,
                enableAutoBranchCreation: props.buildSettings?.enableAutoBranchCreation ?? false,
                enablePullRequestPreview: true,
            },
            basicAuthConfig: props.basicAuth
                ? {
                    enableBasicAuth: props.basicAuth.enableBasicAuth ?? true,
                    username: props.basicAuth.username,
                    password: props.basicAuth.password,
                }
                : undefined,
            tags: props.tags
                ? Object.entries(props.tags).map(([key, value]) => ({ key, value }))
                : undefined,
        });
        // Create branches
        if (props.branches) {
            props.branches.forEach((branch, index) => {
                const cfnBranch = new amplify.CfnBranch(this, `Branch${index}`, {
                    appId: this.app.attrAppId,
                    branchName: branch.branchName,
                    stage: branch.stage || "PRODUCTION",
                    environmentVariables: this.formatEnvVars(branch.environmentVariables),
                    enableAutoBuild: branch.enableAutoBuild ?? true,
                    framework: branch.framework,
                    buildSpec: branch.buildSpec,
                });
                this.branches.push(cfnBranch);
            });
        }
        else {
            // Default main branch
            const mainBranch = new amplify.CfnBranch(this, "MainBranch", {
                appId: this.app.attrAppId,
                branchName: "main",
                stage: "PRODUCTION",
                enableAutoBuild: true,
            });
            this.branches.push(mainBranch);
        }
        // Custom domain setup
        if (props.customDomain) {
            this.domain = new amplify.CfnDomain(this, "Domain", {
                appId: this.app.attrAppId,
                domainName: props.customDomain.domainName,
                enableAutoSubDomain: props.customDomain.enableAutoSubdomain,
                autoSubDomainCreationPatterns: props.customDomain.autoSubdomainCreationPatterns,
                subDomainSettings: props.customDomain.subDomains?.map((sub) => ({
                    branchName: sub.branchName,
                    prefix: sub.prefix,
                })) || [
                    {
                        branchName: "main",
                        prefix: "",
                    },
                ],
            });
        }
        // Outputs
        new aws_cdk_lib_1.CfnOutput(this, "AppId", {
            value: this.app.attrAppId,
            description: "Amplify App ID",
        });
        new aws_cdk_lib_1.CfnOutput(this, "DefaultDomain", {
            value: this.app.attrDefaultDomain,
            description: "Amplify Default Domain",
        });
        if (this.domain) {
            new aws_cdk_lib_1.CfnOutput(this, "CustomDomain", {
                value: props.customDomain.domainName,
                description: "Amplify Custom Domain",
            });
        }
    }
    formatEnvVars(envVars) {
        if (!envVars)
            return undefined;
        return Object.entries(envVars).map(([name, value]) => ({ name, value }));
    }
    getDefaultCustomRules() {
        return [
            {
                source: "/<*>",
                target: "/index.html",
                status: "404-200",
            },
        ];
    }
    getGitHubAuthentication(props) {
        console.log("🔍 Attempting to retrieve GitHub authentication...");
        // Priority 1: Direct access token
        if (props.accessToken) {
            console.log("🔑 Using provided access token...");
            console.log(`🔑 Token length: ${props.accessToken.length} characters`);
            console.log(`🔑 Token prefix: ${props.accessToken.substring(0, 7)}...`);
            return props.accessToken;
        }
        // Priority 2: Fallback to GitHub CLI or environment variable
        console.log("🔄 Falling back to GitHub CLI or environment variable...");
        return this.getGitHubToken();
    }
    getGitHubToken() {
        console.log("🔍 Attempting to retrieve GitHub token...");
        try {
            // Try to get token from gh CLI first
            console.log("📱 Trying to get token from GitHub CLI (gh auth token)...");
            const token = (0, child_process_1.execSync)("gh auth token", {
                encoding: "utf8",
                stdio: "pipe",
            })
                .toString()
                .trim();
            if (token && token.length > 0) {
                console.log("✅ Successfully retrieved GitHub token from gh CLI");
                console.log("💡 Tip: For production deployments, consider migrating to GitHub Apps");
                console.log("   for better security and organization support after setup is complete.");
                return token;
            }
            else {
                console.log("⚠️  gh CLI returned empty token");
            }
        }
        catch (error) {
            console.log("❌ Failed to get token from gh CLI:");
            console.log(`   Error: ${error.message}`);
            console.log("   This might mean:");
            console.log("   - GitHub CLI is not installed");
            console.log("   - You're not authenticated with gh CLI");
            console.log("   - Run 'gh auth login' to authenticate");
        }
        // Fallback to environment variable
        console.log("🔄 Falling back to environment variable MUFIN_PUSH_TOKEN...");
        const envToken = process.env.MUFIN_PUSH_TOKEN;
        if (envToken && envToken.length > 0) {
            console.log("✅ Successfully retrieved GitHub token from environment variable");
            console.log("💡 Tip: For production deployments, consider migrating to GitHub Apps");
            console.log("   for better security and organization support after setup is complete.");
            return envToken;
        }
        else {
            console.log("⚠️  Environment variable MUFIN_PUSH_TOKEN is not set or empty");
        }
        console.log("❌ No GitHub token found! Please either:");
        console.log("   1. Run 'gh auth login' to authenticate with GitHub CLI, or");
        console.log("   2. Set the MUFIN_PUSH_TOKEN environment variable");
        console.log("   3. Pass accessToken directly in AmplifyAppProps");
        console.log("");
        console.log("💡 After setup: Consider GitHub Apps for production deployments");
        return undefined;
    }
    getDefaultBuildSpec(platform) {
        if (platform === "WEB_COMPUTE") {
            return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - yarn install
    build:
      commands:
        - yarn run build
  artifacts:
    baseDirectory: .next
    files:
      - '**/*'
  cache:
    paths:
      - .next/cache/**/*
      - node_modules/**/*`;
        }
        return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - npm ci
    build:
      commands:
        - npm run build
  artifacts:
    baseDirectory: dist
    files:
      - '**/*'
  cache:
    paths:
      - node_modules/**/*`;
    }
}
exports.AmplifyApp = AmplifyApp;
_a = JSII_RTTI_SYMBOL_1;
AmplifyApp[_a] = { fqn: "must-cdk.AmplifyApp", version: "0.0.78" };
//# sourceMappingURL=data:application/json;base64,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