"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.amplifyCommand = void 0;
exports.amplifyInitCommand = amplifyInitCommand;
const fs = require("fs");
const path = require("path");
const commander_1 = require("commander");
/**
 * Create the amplify command with subcommands
 */
exports.amplifyCommand = new commander_1.Command("amplify")
    .description("Amplify-related commands")
    .addCommand(new commander_1.Command("init")
    .description("Initialize a new Amplify project with React application template")
    .option("-d, --directory <dir>", "Target directory", process.cwd())
    .action(amplifyInitCommand));
/**
 * Initialize a new Amplify project by copying the Next.js application template
 */
async function amplifyInitCommand(options) {
    try {
        const { directory } = options;
        // Ensure the directory exists
        if (!fs.existsSync(directory)) {
            console.error(`❌ Directory does not exist: ${directory}`);
            process.exit(1);
        }
        // Get the path to the amplify example
        const examplePath = getAmplifyExamplePath();
        if (!fs.existsSync(examplePath)) {
            console.error(`❌ Amplify example not found at: ${examplePath}`);
            console.error("   Make sure must-cdk is properly installed.");
            console.error("");
            console.error("💡 Troubleshooting:");
            console.error("   • If installed globally: npm install -g must-cdk@latest");
            console.error("   • If installed locally: npm install must-cdk");
            console.error("   • Try reinstalling the package");
            console.error("");
            console.error("📚 For more help, visit: https://github.com/globalmsq/must-cdk");
            process.exit(1);
        }
        console.log(`📁 Initializing Amplify React application in: ${directory}`);
        console.log(`📂 Source: ${examplePath}`);
        console.log("");
        // Copy all files and directories recursively
        const initializedItems = copyDirectoryRecursive(examplePath, directory);
        console.log("");
        console.log(`🎉 Successfully initialized project with ${initializedItems.files} files and ${initializedItems.directories} directories!`);
        console.log("");
        console.log("📝 Files initialized:");
        console.log("   • README.md - Project documentation and setup instructions");
        console.log("   • src/ - React application source code");
        console.log("   • public/ - Static assets and files");
        console.log("   • amplify/ - Amplify configuration files");
        console.log("   • package.json & package-lock.json - Dependencies");
        console.log("");
        console.log("🚀 Next steps:");
        console.log("   1. Review the README.md for setup instructions");
        console.log("   2. Install dependencies: npm install");
        console.log("   3. Configure your Amplify app settings");
        console.log("   4. Deploy using Amplify console or CLI");
        console.log("");
        console.log("📚 For more information, visit:");
        console.log("   https://github.com/globalmsq/must-cdk");
    }
    catch (error) {
        console.error("❌ Error during Amplify initialization:", error);
        process.exit(1);
    }
}
/**
 * Copy a directory and its contents recursively
 */
function copyDirectoryRecursive(source, target) {
    let fileCount = 0;
    let dirCount = 0;
    function copyRecursive(src, dest) {
        const items = fs.readdirSync(src);
        for (const item of items) {
            const srcPath = path.join(src, item);
            const destPath = path.join(dest, item);
            const stat = fs.statSync(srcPath);
            if (stat.isDirectory()) {
                // Skip node_modules and other common directories that shouldn't be copied
                if (item === "node_modules" ||
                    item === ".git" ||
                    item === ".next" ||
                    item === "dist") {
                    continue;
                }
                // Create directory if it doesn't exist
                if (!fs.existsSync(destPath)) {
                    fs.mkdirSync(destPath, { recursive: true });
                    dirCount++;
                }
                // Recursively copy directory contents
                copyRecursive(srcPath, destPath);
            }
            else {
                // Skip hidden files and lock files that might cause issues
                if (item.startsWith(".") &&
                    item !== ".gitignore" &&
                    item !== ".env.example") {
                    continue;
                }
                // Check if file already exists
                if (fs.existsSync(destPath)) {
                    continue;
                }
                // Copy the file
                fs.copyFileSync(srcPath, destPath);
                fileCount++;
            }
        }
    }
    copyRecursive(source, target);
    return { files: fileCount, directories: dirCount };
}
/**
 * Get the path to the amplify example directory
 */
function getAmplifyExamplePath() {
    const currentDir = __dirname;
    // When running from compiled lib/ directory (development)
    let examplePath = path.resolve(currentDir, "../../../src/amplify/example");
    if (fs.existsSync(examplePath)) {
        return examplePath;
    }
    // When running from source src/ directory (development)
    examplePath = path.resolve(currentDir, "../amplify/example");
    if (fs.existsSync(examplePath)) {
        return examplePath;
    }
    // When installed as global package - try to find the package installation directory
    try {
        // Try to resolve the package.json of must-cdk
        const packageJsonPath = require.resolve("must-cdk/package.json");
        const packageDir = path.dirname(packageJsonPath);
        examplePath = path.join(packageDir, "src/amplify/example");
        if (fs.existsSync(examplePath)) {
            return examplePath;
        }
    }
    catch (error) {
        // If require.resolve fails, continue with other attempts
    }
    // Try relative to the CLI script location (for global installs)
    // Global npm packages are typically in: /usr/local/lib/node_modules/must-cdk/
    // or ~/.npm-global/lib/node_modules/must-cdk/
    const possibleGlobalPaths = [
        path.resolve(currentDir, "../../src/amplify/example"), // lib/cli/commands -> src/amplify/example
        path.resolve(currentDir, "../../../src/amplify/example"), // if nested deeper
    ];
    for (const possiblePath of possibleGlobalPaths) {
        if (fs.existsSync(possiblePath)) {
            return possiblePath;
        }
    }
    // Last resort: try relative to process.cwd() (shouldn't normally be needed)
    examplePath = path.resolve(process.cwd(), "src/amplify/example");
    return examplePath;
}
//# sourceMappingURL=data:application/json;base64,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