"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToOrigins = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
class CloudFrontToOrigins extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.aRecords = [];
        this.s3BucketsMap = new Map();
        this.httpOriginsMap = new Map();
        this.s3OriginsMap = new Map();
        // Validate that at least one origin is provided
        if ((!props.s3Origins || props.s3Origins.length === 0) &&
            (!props.httpOrigins || props.httpOrigins.length === 0)) {
            throw new Error("At least one origin (an S3 Bucket or a HTTP domain) must be provided");
        }
        // Apply intelligent defaults
        const enhancedProps = this.applyDefaults(props);
        let certificate;
        let logBucket;
        // Setup all S3 origins
        if (enhancedProps.s3Origins) {
            for (const s3Config of enhancedProps.s3Origins) {
                this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
                const s3Origin = this.setupS3Origin(s3Config);
                this.s3OriginsMap.set(s3Config.id, s3Origin);
            }
        }
        // Setup all HTTP origins
        if (enhancedProps.httpOrigins) {
            for (const httpConfig of enhancedProps.httpOrigins) {
                const httpOrigin = this.setupHttpOrigin(httpConfig);
                this.httpOriginsMap.set(httpConfig.id, httpOrigin);
            }
        }
        // Setup custom domain and certificate
        if (enhancedProps.customDomainName) {
            certificate = this.setupCustomDomain(enhancedProps);
        }
        // Setup logging (enabled by default unless explicitly disabled)
        if (enhancedProps.enableLogging !== false) {
            logBucket = this.setupLogging(enhancedProps);
        }
        // Create CloudFront distribution
        this.distribution = this.createDistribution(enhancedProps, {
            certificate,
            logBucket,
        });
        // Setup Route53 records for all domains
        if (enhancedProps.createRoute53Records !== false &&
            enhancedProps.hostedZone) {
            this.setupRoute53Records(enhancedProps);
        }
        // Assign readonly properties
        this.logBucket = logBucket;
        this.certificate = certificate;
        this.domainNames = this.getAllDomainNames(enhancedProps);
    }
    /**
     * Get all domain names (primary + additional)
     */
    getAllDomainNames(props) {
        const domains = [];
        if (props.customDomainName) {
            domains.push(props.customDomainName);
        }
        if (props.additionalDomainNames) {
            domains.push(...props.additionalDomainNames);
        }
        return domains.length > 0 ? domains : undefined;
    }
    /**
     * Apply intelligent defaults based on origin configuration
     */
    applyDefaults(props) {
        return {
            // Apply base defaults
            enableLogging: true,
            createRoute53Records: true,
            priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
            enabled: true,
            // Override with user props
            ...props,
            // Apply HTTP origin protocol defaults
            httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
                ...httpOrigin,
            })),
            // Merge cache behaviors
            cacheBehaviors: [...(props.cacheBehaviors || [])],
        };
    }
    /**
     * Setup S3 origin using Origin Access Identity (OAI)
     */
    setupS3Origin(s3Config) {
        // Use Origin Access Identity
        const originAccessIdentity = s3Config.originAccessIdentity ||
            new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
                comment: `OAI for ${s3Config.bucket.bucketName}`,
            });
        // Grant CloudFront access to S3 bucket
        s3Config.bucket.grantRead(originAccessIdentity);
        return origins.S3BucketOrigin.withOriginAccessIdentity(s3Config.bucket, {
            originAccessIdentity,
            originPath: s3Config.originPath,
            originId: s3Config.id,
            ...s3Config.s3OriginProps,
        });
    }
    /**
     * Setup HTTP origin
     */
    setupHttpOrigin(httpConfig) {
        return new origins.HttpOrigin(httpConfig.domainName, {
            originPath: httpConfig.originPath,
            protocolPolicy: httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: httpConfig.httpPort,
            httpsPort: httpConfig.httpsPort,
            originId: httpConfig.id,
            ...httpConfig.httpOriginProps,
        });
    }
    /**
     * Setup custom domain and certificate
     */
    setupCustomDomain(props) {
        // If certificate is provided directly, use it
        if (props.certificate) {
            return props.certificate;
        }
        // Auto-create certificate in us-east-1 if hostedZone is provided
        if (props.hostedZone && props.customDomainName) {
            return this.createCrossRegionCertificate(props);
        }
        throw new Error("Either certificate or hostedZone must be provided for custom domain. " +
            "If hostedZone is provided, certificate will be automatically created in us-east-1.");
    }
    /**
     * Create certificate in us-east-1 for CloudFront
     */
    createCrossRegionCertificate(props) {
        const allDomains = this.getAllDomainNames(props) || [
            props.customDomainName,
        ];
        // Use DnsValidatedCertificate to create certificate in us-east-1
        return new acm.DnsValidatedCertificate(this, "Certificate", {
            domainName: props.customDomainName,
            subjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
            hostedZone: props.hostedZone,
            region: "us-east-1",
        });
    }
    /**
     * Setup CloudFront logging
     */
    setupLogging(props) {
        return (props.logBucket ||
            new s3.Bucket(this, "LogBucket", {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                versioned: false,
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(90),
                        transitions: [
                            {
                                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                                transitionAfter: aws_cdk_lib_1.Duration.days(30),
                            },
                        ],
                    },
                ],
            }));
    }
    /**
     * Get origin by ID
     */
    getOriginById(originId) {
        // Check S3 origins first
        const s3Origin = this.s3OriginsMap.get(originId);
        if (s3Origin) {
            return s3Origin;
        }
        // Check HTTP origins
        const httpOrigin = this.httpOriginsMap.get(originId);
        if (httpOrigin) {
            return httpOrigin;
        }
        throw new Error(`Origin with ID '${originId}' not found`);
    }
    /**
     * Determine default origin ID
     */
    getDefaultOriginId(props) {
        if (props.defaultOriginId) {
            return props.defaultOriginId;
        }
        // Default to first HTTP origin, then first S3 origin
        if (props.httpOrigins && props.httpOrigins.length > 0) {
            return props.httpOrigins[0].id;
        }
        if (props.s3Origins && props.s3Origins.length > 0) {
            return props.s3Origins[0].id;
        }
        throw new Error("No origins configured");
    }
    /**
     * Create CloudFront distribution
     */
    createDistribution(props, resources) {
        // Determine default origin
        const defaultOriginId = this.getDefaultOriginId(props);
        const defaultOrigin = this.getOriginById(defaultOriginId);
        const defaultBehavior = {
            origin: defaultOrigin,
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            compress: true,
        };
        // Build additional behaviors
        const additionalBehaviors = {};
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                const origin = this.getOriginById(behavior.originId);
                // Build behavior options object properly
                const behaviorConfig = {
                    origin,
                    viewerProtocolPolicy: behavior.viewerProtocolPolicy ||
                        cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    allowedMethods: behavior.allowedMethods,
                    cachedMethods: behavior.cachedMethods,
                    compress: behavior.compress ?? true,
                };
                // Add cache policy (prioritize direct policy over ID)
                if (behavior.cachePolicy) {
                    behaviorConfig.cachePolicy = behavior.cachePolicy;
                }
                else if (behavior.cachePolicyId) {
                    behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(this, `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.cachePolicyId);
                }
                // Add origin request policy
                if (behavior.originRequestPolicy) {
                    behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
                }
                else if (behavior.originRequestPolicyId) {
                    behaviorConfig.originRequestPolicy =
                        cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(this, `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.originRequestPolicyId);
                }
                // Add response headers policy
                if (behavior.responseHeadersPolicy) {
                    behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
                }
                else if (behavior.responseHeadersPolicyId) {
                    behaviorConfig.responseHeadersPolicy =
                        cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(this, `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.responseHeadersPolicyId);
                }
                additionalBehaviors[behavior.pathPattern] =
                    behaviorConfig;
            }
        }
        // Ensure all origins are referenced by creating behaviors for unused origins
        const usedOriginIds = new Set([defaultOriginId]);
        // Track origins used in cache behaviors
        if (props.cacheBehaviors) {
            props.cacheBehaviors.forEach((behavior) => {
                usedOriginIds.add(behavior.originId);
            });
        }
        // Create default behaviors for unused origins
        const allOriginIds = [
            ...this.s3OriginsMap.keys(),
            ...this.httpOriginsMap.keys(),
        ];
        allOriginIds.forEach((originId) => {
            if (!usedOriginIds.has(originId)) {
                const origin = this.getOriginById(originId);
                // Create a unique path pattern for this origin (lowercase)
                const pathPattern = `/${originId.toLowerCase()}/*`;
                additionalBehaviors[pathPattern] = {
                    origin,
                    viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    compress: true,
                };
            }
        });
        // Prepare distribution props
        const distributionProps = {
            comment: props.comment || `CloudFront distribution for ${this.node.id}`,
            defaultBehavior,
            additionalBehaviors,
            domainNames: this.getAllDomainNames(props),
            certificate: resources.certificate,
            defaultRootObject: props.defaultRootObject,
            enabled: props.enabled,
            priceClass: props.priceClass,
            geoRestriction: props.geoRestriction,
            enableLogging: props.enableLogging !== false,
            logBucket: resources.logBucket,
            logFilePrefix: props.logPrefix,
            logIncludesCookies: props.logIncludeCookies,
            errorResponses: props.errorPages,
            webAclId: props.webAclId,
            enableIpv6: props.enableIpv6,
            httpVersion: props.httpVersion,
        };
        return new cloudfront.Distribution(this, "Distribution", distributionProps);
    }
    /**
     * Setup Route53 alias records for all domains
     */
    setupRoute53Records(props) {
        if (!props.hostedZone) {
            return;
        }
        const domains = this.getAllDomainNames(props);
        if (!domains) {
            return;
        }
        // Create A record for each domain
        domains.forEach((domain, index) => {
            const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;
            const aRecord = new route53.ARecord(this, recordId, {
                zone: props.hostedZone,
                recordName: domain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
            });
            this.aRecords.push(aRecord);
        });
    }
    /**
     * Get S3 bucket by origin ID
     */
    getS3Bucket(originId) {
        return this.s3BucketsMap.get(originId);
    }
    /**
     * Get HTTP origin by origin ID
     */
    getHttpOrigin(originId) {
        return this.httpOriginsMap.get(originId);
    }
    /**
     * Get all S3 bucket origin IDs
     */
    get s3OriginIds() {
        return Array.from(this.s3BucketsMap.keys());
    }
    /**
     * Get all HTTP origin IDs
     */
    get httpOriginIds() {
        return Array.from(this.httpOriginsMap.keys());
    }
    /**
     * Get all S3 buckets as an array of objects with ID and bucket
     */
    get s3Origins() {
        return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
            id,
            bucket,
        }));
    }
    /**
     * Get all HTTP origins as an array of objects with ID and origin
     */
    get httpOrigins() {
        return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
            id,
            origin,
        }));
    }
    /**
     * Get the CloudFront distribution domain name
     */
    get distributionDomainName() {
        return this.distribution.distributionDomainName;
    }
    /**
     * Get the CloudFront distribution URL with protocol
     */
    get distributionUrl() {
        return `https://${this.distribution.distributionDomainName}`;
    }
    /**
     * Get the custom domain URL (if configured)
     */
    get customDomainUrl() {
        return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
    }
}
exports.CloudFrontToOrigins = CloudFrontToOrigins;
_a = JSII_RTTI_SYMBOL_1;
CloudFrontToOrigins[_a] = { fqn: "must-cdk.CloudFrontToOrigins", version: "0.0.78" };
//# sourceMappingURL=data:application/json;base64,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