#!/usr/bin/env node

const fs = require('fs');
const path = require('path');

/**
 * Script to split Python API documentation into separate files based on level 3 headings (construct names)
 * Groups each construct with its related properties and interfaces
 * Saves directly to docs/python folder
 */
function splitApiDocs() {
  const apiFilePath = path.join(__dirname, '..', 'docs', 'python', 'api.md');
  const outputDir = path.join(__dirname, '..', 'docs', 'python');

  // Check if Python API.md exists
  if (!fs.existsSync(apiFilePath)) {
    console.error('Python API documentation not found at docs/python/api.md. Please run "npx projen docs:python" first.');
    process.exit(1);
  }

  // Read the Python API.md file
  const content = fs.readFileSync(apiFilePath, 'utf8');
  
  // Split content by level 3 headings (### ConstructName)
  const sections = content.split(/(?=^### \w+)/m);
  
  // Keep the header section (everything before first ### heading)
  const headerSection = sections[0];
  
  // Group constructs with their related properties
  const constructGroups = groupConstructsWithProps(sections.slice(1));
  
  // Process each construct group and save to python directory
  constructGroups.forEach(group => {
    const constructName = group.mainConstruct;
    
    // Combine all sections for this construct group
    let combinedContent = headerSection + group.sections.join('');
    
    // Fix cross-references to point to the correct files
    combinedContent = fixCrossReferences(combinedContent, constructGroups, group);
    
    // Write to python directory
    const outputFile = path.join(outputDir, `${constructName}.md`);
    fs.writeFileSync(outputFile, combinedContent);
    
    console.log(`Created: docs/python/${constructName}.md (includes ${group.relatedProps.length} related props)`);
  });

  // Create index file in python directory
  createIndexFile(outputDir, constructGroups);
  
  console.log(`\nPython API documentation split into ${constructGroups.length} construct files.`);
  console.log(`Files saved to: docs/python/`);
}

/**
 * Group constructs with their related properties/interfaces
 */
function groupConstructsWithProps(sections) {
  const constructs = [];
  const props = [];
  
  // Separate constructs from props/interfaces
  sections.forEach(section => {
    const headingMatch = section.match(/^### (\w+)/);
    if (!headingMatch) return;
    
    const name = headingMatch[1];
    
    // Check if it's a construct (has Initializers section) or a props/interface
    if (section.includes('#### Initializers') || section.includes('#### Methods')) {
      constructs.push({ name, section });
    } else {
      props.push({ name, section });
    }
  });
  
  // Group props with their related constructs
  const groups = constructs.map(construct => {
    const constructName = construct.name;
    const relatedProps = [];
    const sections = [construct.section];
    
    // Find related props (those that start with the construct name or are referenced)
    props.forEach(prop => {
      if (isRelatedProp(constructName, prop.name, construct.section)) {
        relatedProps.push(prop.name);
        sections.push(prop.section);
      }
    });
    
    return {
      mainConstruct: constructName,
      relatedProps,
      sections
    };
  });
  
  // Add any remaining props that weren't grouped
  const groupedPropNames = new Set();
  groups.forEach(group => {
    group.relatedProps.forEach(prop => groupedPropNames.add(prop));
  });
  
  const ungroupedProps = props.filter(prop => !groupedPropNames.has(prop.name));
  ungroupedProps.forEach(prop => {
    groups.push({
      mainConstruct: prop.name,
      relatedProps: [],
      sections: [prop.section]
    });
  });
  
  return groups;
}

/**
 * Determine if a prop is related to a construct
 */
function isRelatedProp(constructName, propName, constructSection) {
  // Direct naming relationship (e.g., AmplifyApp -> AmplifyAppProps)
  if (propName.startsWith(constructName)) {
    return true;
  }
  
  // Check if the prop is referenced in the construct section
  if (constructSection.includes(propName)) {
    return true;
  }
  
  // Common patterns
  const commonPatterns = [
    `${constructName}Props`,
    `${constructName}Config`,
    `${constructName}Options`,
    `${constructName}Settings`
  ];
  
  return commonPatterns.includes(propName);
}

/**
 * Fix cross-references in the content to point to correct files
 */
function fixCrossReferences(content, allGroups, currentGroup) {
  // Create a map of construct/interface names to their file names
  const nameToFile = {};
  allGroups.forEach(group => {
    nameToFile[group.mainConstruct] = `${group.mainConstruct}.md`;
    group.relatedProps.forEach(prop => {
      nameToFile[prop] = `${group.mainConstruct}.md`;
    });
  });

  // Fix all cross-references
  return content.replace(/\[<a href="#must-cdk\.(\w+)">([^<]+)<\/a>\]/g, (match, referenceName, displayText) => {
    if (nameToFile[referenceName]) {
      // If it's in the current file, use local anchor
      if (nameToFile[referenceName] === `${currentGroup.mainConstruct}.md`) {
        return `[${displayText}](#must-cdk.${referenceName})`;
      } else {
        // If it's in another file, link to that file
        return `[${displayText}](./${nameToFile[referenceName]}#must-cdk.${referenceName})`;
      }
    }
    // If not found, keep original link
    return match;
  });
}

/**
 * Create an index file listing all constructs
 */
function createIndexFile(outputDir, groups) {
  const indexContent = `# Must CDK Constructs

This directory contains individual documentation files for each construct in the Must CDK library.

## Available Constructs

${groups
  .filter(group => group.sections[0].includes('#### Initializers'))
  .map(group => `- [${group.mainConstruct}](./${group.mainConstruct}.md)${group.relatedProps.length > 0 ? ` (includes ${group.relatedProps.join(', ')})` : ''}`)
  .join('\n')}

## Interfaces and Types

${groups
  .filter(group => !group.sections[0].includes('#### Initializers'))
  .map(group => `- [${group.mainConstruct}](./${group.mainConstruct}.md)`)
  .join('\n')}

## Usage

Each construct file contains:
- Constructor parameters and types
- Available methods and properties
- Related interfaces and type definitions
- Usage examples where applicable

For the complete API reference, see the main [API.md](./api.md) file.
`;

  const indexFile = path.join(outputDir, 'constructs-README.md');
  fs.writeFileSync(indexFile, indexContent);
  console.log(`Created: docs/python/constructs-README.md`);
}

// Run the script
if (require.main === module) {
  splitApiDocs();
}

module.exports = { splitApiDocs };
