import { execSync } from "child_process";
import { CfnOutput } from "aws-cdk-lib";
import * as amplify from "aws-cdk-lib/aws-amplify";
import { Construct } from "constructs";
import { AmplifyAppProps, CustomRule } from "./interface";

export class AmplifyApp extends Construct {
  public readonly app: amplify.CfnApp;
  public readonly branches: amplify.CfnBranch[] = [];
  public readonly domain?: amplify.CfnDomain;

  constructor(scope: Construct, id: string, props: AmplifyAppProps) {
    super(scope, id);

    // Get GitHub authentication
    const accessToken = this.getGitHubAuthentication(props);

    if (!accessToken) {
      throw new Error(`
❌ GitHub authentication is required for Amplify App deployment!

Please provide authentication using one of these methods:

1. 🔑 Personal Access Token (Current approach):
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: 'your-github-token'
   })

2. 📱 Use GitHub CLI (for development):
   gh auth login
   
3. 🌍 Set environment variable:
   export MUFIN_PUSH_TOKEN=your-github-token

4. 🔒 Use AWS Secrets Manager:
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: SecretValue.secretsManager('github-token').unsafeUnwrap()
   })

💡 Production Recommendation:
   After your project is set up and working, consider migrating to GitHub Apps
   for better security, organization support, and higher rate limits.
   See: https://docs.github.com/en/apps/creating-github-apps
      `);
    }

    console.log("🚀 Creating Amplify App with authenticated GitHub access...");

    // Create Amplify App
    this.app = new amplify.CfnApp(this, "App", {
      name: props.appName,
      repository: props.repository,
      accessToken: accessToken,
      environmentVariables: this.formatEnvVars(props.environmentVariables),
      buildSpec: props.buildSpec || this.getDefaultBuildSpec(props.platform),
      platform: props.platform || "WEB_COMPUTE",
      customRules: props.customRules || this.getDefaultCustomRules(),
      autoBranchCreationConfig: {
        enableAutoBuild: props.buildSettings?.enableBranchAutoBuild ?? false,
        enableAutoBranchCreation:
          props.buildSettings?.enableAutoBranchCreation ?? false,
        enablePullRequestPreview: true,
      },
      basicAuthConfig: props.basicAuth
        ? {
            enableBasicAuth: props.basicAuth.enableBasicAuth ?? true,
            username: props.basicAuth.username,
            password: props.basicAuth.password,
          }
        : undefined,
      tags: props.tags
        ? Object.entries(props.tags).map(([key, value]) => ({ key, value }))
        : undefined,
    });

    // Create branches
    if (props.branches) {
      props.branches.forEach((branch, index) => {
        const cfnBranch = new amplify.CfnBranch(this, `Branch${index}`, {
          appId: this.app.attrAppId,
          branchName: branch.branchName,
          stage: branch.stage || "PRODUCTION",
          environmentVariables: this.formatEnvVars(branch.environmentVariables),
          enableAutoBuild: branch.enableAutoBuild ?? true,
          framework: branch.framework,
          buildSpec: branch.buildSpec,
        });
        this.branches.push(cfnBranch);
      });
    } else {
      // Default main branch
      const mainBranch = new amplify.CfnBranch(this, "MainBranch", {
        appId: this.app.attrAppId,
        branchName: "main",
        stage: "PRODUCTION",
        enableAutoBuild: true,
      });
      this.branches.push(mainBranch);
    }

    // Custom domain setup
    if (props.customDomain) {
      this.domain = new amplify.CfnDomain(this, "Domain", {
        appId: this.app.attrAppId,
        domainName: props.customDomain.domainName,
        enableAutoSubDomain: props.customDomain.enableAutoSubdomain,
        autoSubDomainCreationPatterns:
          props.customDomain.autoSubdomainCreationPatterns,
        subDomainSettings: props.customDomain.subDomains?.map((sub) => ({
          branchName: sub.branchName,
          prefix: sub.prefix,
        })) || [
          {
            branchName: "main",
            prefix: "",
          },
        ],
      });
    }

    // Outputs
    new CfnOutput(this, "AppId", {
      value: this.app.attrAppId,
      description: "Amplify App ID",
    });

    new CfnOutput(this, "DefaultDomain", {
      value: this.app.attrDefaultDomain,
      description: "Amplify Default Domain",
    });

    if (this.domain) {
      new CfnOutput(this, "CustomDomain", {
        value: props.customDomain!.domainName,
        description: "Amplify Custom Domain",
      });
    }
  }

  private formatEnvVars(
    envVars?: Record<string, string>,
  ): amplify.CfnApp.EnvironmentVariableProperty[] | undefined {
    if (!envVars) return undefined;
    return Object.entries(envVars).map(([name, value]) => ({ name, value }));
  }

  private getDefaultCustomRules(): CustomRule[] {
    return [
      {
        source: "/<*>",
        target: "/index.html",
        status: "404-200",
      },
    ];
  }

  private getGitHubAuthentication(props: AmplifyAppProps): string | undefined {
    console.log("🔍 Attempting to retrieve GitHub authentication...");

    // Priority 1: Direct access token
    if (props.accessToken) {
      console.log("🔑 Using provided access token...");
      console.log(`🔑 Token length: ${props.accessToken.length} characters`);
      console.log(`🔑 Token prefix: ${props.accessToken.substring(0, 7)}...`);
      return props.accessToken;
    }

    // Priority 2: Fallback to GitHub CLI or environment variable
    console.log("🔄 Falling back to GitHub CLI or environment variable...");
    return this.getGitHubToken();
  }

  private getGitHubToken(): string | undefined {
    console.log("🔍 Attempting to retrieve GitHub token...");

    try {
      // Try to get token from gh CLI first
      console.log("📱 Trying to get token from GitHub CLI (gh auth token)...");
      const token = execSync("gh auth token", {
        encoding: "utf8",
        stdio: "pipe",
      })
        .toString()
        .trim();

      if (token && token.length > 0) {
        console.log("✅ Successfully retrieved GitHub token from gh CLI");
        console.log(
          "💡 Tip: For production deployments, consider migrating to GitHub Apps",
        );
        console.log(
          "   for better security and organization support after setup is complete.",
        );
        return token;
      } else {
        console.log("⚠️  gh CLI returned empty token");
      }
    } catch (error: any) {
      console.log("❌ Failed to get token from gh CLI:");
      console.log(`   Error: ${error.message}`);
      console.log("   This might mean:");
      console.log("   - GitHub CLI is not installed");
      console.log("   - You're not authenticated with gh CLI");
      console.log("   - Run 'gh auth login' to authenticate");
    }

    // Fallback to environment variable
    console.log("🔄 Falling back to environment variable MUFIN_PUSH_TOKEN...");
    const envToken = process.env.MUFIN_PUSH_TOKEN;

    if (envToken && envToken.length > 0) {
      console.log(
        "✅ Successfully retrieved GitHub token from environment variable",
      );
      console.log(
        "💡 Tip: For production deployments, consider migrating to GitHub Apps",
      );
      console.log(
        "   for better security and organization support after setup is complete.",
      );
      return envToken;
    } else {
      console.log(
        "⚠️  Environment variable MUFIN_PUSH_TOKEN is not set or empty",
      );
    }

    console.log("❌ No GitHub token found! Please either:");
    console.log(
      "   1. Run 'gh auth login' to authenticate with GitHub CLI, or",
    );
    console.log("   2. Set the MUFIN_PUSH_TOKEN environment variable");
    console.log("   3. Pass accessToken directly in AmplifyAppProps");
    console.log("");
    console.log(
      "💡 After setup: Consider GitHub Apps for production deployments",
    );

    return undefined;
  }

  private getDefaultBuildSpec(platform?: string): string {
    if (platform === "WEB_COMPUTE") {
      return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - yarn install
    build:
      commands:
        - yarn run build
  artifacts:
    baseDirectory: .next
    files:
      - '**/*'
  cache:
    paths:
      - .next/cache/**/*
      - node_modules/**/*`;
    }

    return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - npm ci
    build:
      commands:
        - npm run build
  artifacts:
    baseDirectory: dist
    files:
      - '**/*'
  cache:
    paths:
      - node_modules/**/*`;
  }
}
