import * as api from "aws-cdk-lib/aws-apigateway";
import * as acm from "aws-cdk-lib/aws-certificatemanager";
import * as lambda from "aws-cdk-lib/aws-lambda";
import * as logs from "aws-cdk-lib/aws-logs";
import * as route53 from "aws-cdk-lib/aws-route53";
import * as targets from "aws-cdk-lib/aws-route53-targets";
import { Construct } from "constructs";
import { ApiGatewayToLambdaProps, CustomRoute } from "./rest-interface";
import { setupLogging } from "./utils";

export class ApiGatewayToLambda extends Construct {
  public readonly apiGateway: api.RestApi;
  public readonly apiGatewayLogGroup?: logs.LogGroup;
  public readonly lambdaFunction: lambda.IFunction;
  public readonly aRecord?: route53.ARecord;
  public readonly certificate?: acm.ICertificate;
  public readonly domain?: api.DomainName;
  public readonly usagePlan?: api.UsagePlan;

  constructor(scope: Construct, id: string, props: ApiGatewayToLambdaProps) {
    super(scope, id);

    this.lambdaFunction = props.lambdaFunction;

    // Determine if we should use proxy mode
    const useProxy = props.proxy !== false && !props.customRoutes;

    if (useProxy) {
      this.apiGateway = new api.LambdaRestApi(this, "ApiGateway", {
        handler: props.lambdaFunction,
        restApiName: props.apiName,
        proxy: true,
        ...props.lambdaApiProps,
      });
    } else {
      this.apiGateway = new api.RestApi(this, "ApiGateway", {
        restApiName: props.apiName,
        ...props.restApiProps,
      });

      // Add custom routes if provided
      if (props.customRoutes) {
        this.addCustomRoutes(props.customRoutes);
      }

      // If no custom routes but proxy is disabled, add default integration
      if (!props.customRoutes && props.proxy === false) {
        this.apiGateway.root.addMethod(
          "ANY",
          new api.LambdaIntegration(props.lambdaFunction),
        );
      }
    }

    // Setup CloudWatch logging
    if (props.enableLogging) {
      const loggingResources = setupLogging(
        this,
        props.apiName,
        props.logGroupProps,
      );
      this.apiGatewayLogGroup = loggingResources.logGroup;
    }

    // Setup custom domain
    if (props.customDomainName) {
      const domainResources = this.setupCustomDomain(props);
      this.certificate = domainResources.certificate;
      this.domain = domainResources.domain;
      this.aRecord = domainResources.aRecord;
    }

    // Setup usage plan
    if (props.createUsagePlan) {
      this.usagePlan = this.setupUsagePlan();
    }
  }

  /**
   * Add custom routes to the API
   */
  private addCustomRoutes(routes: CustomRoute[]) {
    const resourceMap = new Map<string, api.IResource>();

    routes.forEach((route) => {
      const pathParts = route.path
        .split("/")
        .filter((part: string) => part !== "");
      let currentResource: api.IResource = this.apiGateway.root;

      // Build nested resources
      let currentPath = "";
      for (const part of pathParts) {
        currentPath += `/${part}`;

        if (!resourceMap.has(currentPath)) {
          currentResource = currentResource.addResource(part);
          resourceMap.set(currentPath, currentResource);
        } else {
          currentResource = resourceMap.get(currentPath)!;
        }
      }

      // Add method to the final resource
      currentResource.addMethod(
        route.method,
        new api.LambdaIntegration(route.handler),
        route.methodOptions,
      );
    });
  }

  /**
   * Setup custom domain with certificate and Route53 record
   */
  private setupCustomDomain(props: ApiGatewayToLambdaProps): {
    certificate: acm.ICertificate;
    domain: api.DomainName;
    aRecord?: route53.ARecord;
  } {
    let certificate: acm.ICertificate;

    // Use existing certificate or create new one
    if (props.existingCertificate) {
      certificate = props.existingCertificate;
    } else if (props.hostedZone) {
      certificate = new acm.Certificate(this, "Certificate", {
        domainName: props.customDomainName!,
        validation: acm.CertificateValidation.fromDns(props.hostedZone),
      });
    } else {
      throw new Error(
        "Either certificateArn or hostedZone must be provided for custom domain",
      );
    }

    // Create custom domain
    const domain = new api.DomainName(this, "CustomDomain", {
      domainName: props.customDomainName!,
      certificate: certificate,
    });

    // Create base path mapping
    new api.BasePathMapping(this, "BasePathMapping", {
      domainName: domain,
      restApi: this.apiGateway,
    });

    let aRecord: route53.ARecord | undefined;

    // Create Route53 alias record if hosted zone provided
    if (props.hostedZone) {
      aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
        zone: props.hostedZone,
        recordName: props.customDomainName!,
        target: route53.RecordTarget.fromAlias(
          new targets.ApiGatewayDomain(domain),
        ),
      });
    }

    return { certificate, domain, aRecord };
  }

  /**
   * Setup usage plan
   */
  private setupUsagePlan(): api.UsagePlan {
    return new api.UsagePlan(this, "UsagePlan", {
      name: `${this.apiGateway.restApiName}-usage-plan`,
      apiStages: [
        {
          api: this.apiGateway,
          stage: this.apiGateway.deploymentStage,
        },
      ],
    });
  }

  /**
   * Add a custom route after construction (for dynamic route addition)
   */
  public addRoute(route: CustomRoute): api.Method {
    const pathParts = route.path
      .split("/")
      .filter((part: string) => part !== "");
    let currentResource: api.IResource = this.apiGateway.root;

    // Navigate/create resource path
    for (const part of pathParts) {
      const existingResource = currentResource.getResource(part);
      if (existingResource) {
        currentResource = existingResource;
      } else {
        currentResource = currentResource.addResource(part);
      }
    }

    return currentResource.addMethod(
      route.method,
      new api.LambdaIntegration(route.handler),
      route.methodOptions,
    );
  }

  public get apiUrl(): string {
    return this.domain?.domainNameAliasDomainName ?? this.apiGateway.url;
  }
}
