import * as apigatewayv2 from "aws-cdk-lib/aws-apigatewayv2";
import * as apigatewayv2_integrations from "aws-cdk-lib/aws-apigatewayv2-integrations";
import * as acm from "aws-cdk-lib/aws-certificatemanager";
import * as lambda from "aws-cdk-lib/aws-lambda";
import * as logs from "aws-cdk-lib/aws-logs";
import * as route53 from "aws-cdk-lib/aws-route53";
import * as targets from "aws-cdk-lib/aws-route53-targets";
import { Construct } from "constructs";
import { setupLogging } from "./utils";
import {
  WebSocketApiGatewayToLambdaProps,
  WebSocketRoute,
} from "./websocket-interface";

export class WebSocketApiGatewayToLambda extends Construct {
  public readonly webSocketApi: apigatewayv2.WebSocketApi;
  public readonly webSocketStage: apigatewayv2.WebSocketStage;
  public readonly apiGatewayLogGroup?: logs.LogGroup;
  public readonly lambdaFunction: lambda.IFunction;
  public readonly aRecord?: route53.ARecord;
  public readonly certificate?: acm.ICertificate;
  public readonly domain?: apigatewayv2.DomainName;

  constructor(
    scope: Construct,
    id: string,
    props: WebSocketApiGatewayToLambdaProps,
  ) {
    super(scope, id);

    this.lambdaFunction = props.lambdaFunction;

    // Create WebSocket API
    this.webSocketApi = new apigatewayv2.WebSocketApi(this, "WebSocketApi", {
      apiName: props.apiName,
      description: `WebSocket API for ${props.apiName}`,
      ...props.apiProps,
    });

    // Create default integration for the primary lambda
    const defaultIntegration =
      new apigatewayv2_integrations.WebSocketLambdaIntegration(
        "DefaultIntegration",
        this.lambdaFunction,
      );

    // Add default route
    this.webSocketApi.addRoute("$default", {
      integration: defaultIntegration,
    });

    // Add custom routes if provided
    if (props.customRoutes) {
      this.addCustomRoutes(props.customRoutes);
    }

    // Create stage
    const stageName = props.stageName || "dev";
    this.webSocketStage = new apigatewayv2.WebSocketStage(
      this,
      "WebSocketStage",
      {
        webSocketApi: this.webSocketApi,
        stageName: stageName,
        autoDeploy: true,
      },
    );

    // Setup CloudWatch logging
    if (props.enableLogging) {
      const loggingResources = setupLogging(
        this,
        props.apiName,
        props.logGroupProps,
      );
      this.apiGatewayLogGroup = loggingResources.logGroup;
    }

    // Setup custom domain
    if (props.customDomainName) {
      const domainResources = this.setupCustomDomain(props);
      this.certificate = domainResources.certificate;
      this.domain = domainResources.domain;
      this.aRecord = domainResources.aRecord;
    }
  }

  /**
   * Add custom routes to the WebSocket API
   */
  private addCustomRoutes(routes: WebSocketRoute[]) {
    routes.forEach((route) => {
      const integration =
        new apigatewayv2_integrations.WebSocketLambdaIntegration(
          `${route.routeKey}Integration`,
          route.handler,
        );

      this.webSocketApi.addRoute(route.routeKey, {
        integration: integration,
        returnResponse: route.routeResponseSelectionExpression !== undefined,
      });
    });
  }

  /**
   * Setup custom domain with certificate and Route53 record
   */
  private setupCustomDomain(props: WebSocketApiGatewayToLambdaProps): {
    certificate: acm.ICertificate;
    domain: apigatewayv2.DomainName;
    aRecord?: route53.ARecord;
  } {
    let certificate: acm.ICertificate;

    // Use existing certificate or create new one
    if (props.existingCertificate) {
      certificate = props.existingCertificate;
    } else if (props.hostedZone) {
      certificate = new acm.Certificate(this, "Certificate", {
        domainName: props.customDomainName!,
        validation: acm.CertificateValidation.fromDns(props.hostedZone),
      });
    } else {
      throw new Error(
        "Either certificateArn or hostedZone must be provided for custom domain",
      );
    }

    // Create custom domain for WebSocket API
    const domain = new apigatewayv2.DomainName(this, "CustomDomain", {
      domainName: props.customDomainName!,
      certificate: certificate,
    });

    // Create API mapping
    new apigatewayv2.ApiMapping(this, "ApiMapping", {
      api: this.webSocketApi,
      domainName: domain,
      stage: this.webSocketStage,
    });

    let aRecord: route53.ARecord | undefined;

    // Create Route53 alias record if hosted zone provided
    if (props.hostedZone) {
      aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
        zone: props.hostedZone,
        recordName: props.customDomainName!,
        target: route53.RecordTarget.fromAlias(
          new targets.ApiGatewayv2DomainProperties(
            domain.regionalDomainName,
            domain.regionalHostedZoneId,
          ),
        ),
      });
    }

    return { certificate, domain, aRecord };
  }

  /**
   * Add a custom route after construction (for dynamic route addition)
   */
  public addRoute(route: WebSocketRoute): apigatewayv2.WebSocketRoute {
    const integration =
      new apigatewayv2_integrations.WebSocketLambdaIntegration(
        `${route.routeKey}Integration`,
        route.handler,
      );

    return this.webSocketApi.addRoute(route.routeKey, {
      integration: integration,
      returnResponse: route.routeResponseSelectionExpression !== undefined,
    });
  }

  /**
   * Get the WebSocket API URL (useful for outputs)
   */
  public get webSocketUrl(): string {
    if (this.domain) {
      return `wss://${this.domain.name}`;
    }
    return this.webSocketStage.url;
  }
}
