import * as fs from "fs";
import * as path from "path";
import { Command } from "commander";

/**
 * Options for the amplify init command
 */
export interface AmplifyInitOptions {
  directory: string;
}

/**
 * Create the amplify command with subcommands
 */
export const amplifyCommand = new Command("amplify")
  .description("Amplify-related commands")
  .addCommand(
    new Command("init")
      .description(
        "Initialize a new Amplify project with React application template",
      )
      .option("-d, --directory <dir>", "Target directory", process.cwd())
      .action(amplifyInitCommand),
  );

/**
 * Initialize a new Amplify project by copying the Next.js application template
 */
export async function amplifyInitCommand(
  options: AmplifyInitOptions,
): Promise<void> {
  try {
    const { directory } = options;

    // Ensure the directory exists
    if (!fs.existsSync(directory)) {
      console.error(`❌ Directory does not exist: ${directory}`);
      process.exit(1);
    }

    // Get the path to the amplify example
    const examplePath = getAmplifyExamplePath();

    if (!fs.existsSync(examplePath)) {
      console.error(`❌ Amplify example not found at: ${examplePath}`);
      console.error("   Make sure must-cdk is properly installed.");
      console.error("");
      console.error("💡 Troubleshooting:");
      console.error(
        "   • If installed globally: npm install -g must-cdk@latest",
      );
      console.error("   • If installed locally: npm install must-cdk");
      console.error("   • Try reinstalling the package");
      console.error("");
      console.error(
        "📚 For more help, visit: https://github.com/globalmsq/must-cdk",
      );
      process.exit(1);
    }

    console.log(`📁 Initializing Amplify React application in: ${directory}`);
    console.log(`📂 Source: ${examplePath}`);
    console.log("");

    // Copy all files and directories recursively
    const initializedItems = copyDirectoryRecursive(examplePath, directory);

    console.log("");
    console.log(
      `🎉 Successfully initialized project with ${initializedItems.files} files and ${initializedItems.directories} directories!`,
    );

    console.log("");
    console.log("📝 Files initialized:");
    console.log(
      "   • README.md - Project documentation and setup instructions",
    );
    console.log("   • src/ - React application source code");
    console.log("   • public/ - Static assets and files");
    console.log("   • amplify/ - Amplify configuration files");
    console.log("   • package.json & package-lock.json - Dependencies");
    console.log("");
    console.log("🚀 Next steps:");
    console.log("   1. Review the README.md for setup instructions");
    console.log("   2. Install dependencies: npm install");
    console.log("   3. Configure your Amplify app settings");
    console.log("   4. Deploy using Amplify console or CLI");
    console.log("");
    console.log("📚 For more information, visit:");
    console.log("   https://github.com/globalmsq/must-cdk");
  } catch (error) {
    console.error("❌ Error during Amplify initialization:", error);
    process.exit(1);
  }
}

/**
 * Copy a directory and its contents recursively
 */
function copyDirectoryRecursive(
  source: string,
  target: string,
): { files: number; directories: number } {
  let fileCount = 0;
  let dirCount = 0;

  function copyRecursive(src: string, dest: string) {
    const items = fs.readdirSync(src);

    for (const item of items) {
      const srcPath = path.join(src, item);
      const destPath = path.join(dest, item);
      const stat = fs.statSync(srcPath);

      if (stat.isDirectory()) {
        // Skip node_modules and other common directories that shouldn't be copied
        if (
          item === "node_modules" ||
          item === ".git" ||
          item === ".next" ||
          item === "dist"
        ) {
          continue;
        }

        // Create directory if it doesn't exist
        if (!fs.existsSync(destPath)) {
          fs.mkdirSync(destPath, { recursive: true });
          dirCount++;
        }

        // Recursively copy directory contents
        copyRecursive(srcPath, destPath);
      } else {
        // Skip hidden files and lock files that might cause issues
        if (
          item.startsWith(".") &&
          item !== ".gitignore" &&
          item !== ".env.example"
        ) {
          continue;
        }

        // Check if file already exists
        if (fs.existsSync(destPath)) {
          continue;
        }

        // Copy the file
        fs.copyFileSync(srcPath, destPath);
        fileCount++;
      }
    }
  }

  copyRecursive(source, target);
  return { files: fileCount, directories: dirCount };
}

/**
 * Get the path to the amplify example directory
 */
function getAmplifyExamplePath(): string {
  const currentDir = __dirname;

  // When running from compiled lib/ directory (development)
  let examplePath = path.resolve(currentDir, "../../../src/amplify/example");
  if (fs.existsSync(examplePath)) {
    return examplePath;
  }

  // When running from source src/ directory (development)
  examplePath = path.resolve(currentDir, "../amplify/example");
  if (fs.existsSync(examplePath)) {
    return examplePath;
  }

  // When installed as global package - try to find the package installation directory
  try {
    // Try to resolve the package.json of must-cdk
    const packageJsonPath = require.resolve("must-cdk/package.json");
    const packageDir = path.dirname(packageJsonPath);
    examplePath = path.join(packageDir, "src/amplify/example");

    if (fs.existsSync(examplePath)) {
      return examplePath;
    }
  } catch (error) {
    // If require.resolve fails, continue with other attempts
  }

  // Try relative to the CLI script location (for global installs)
  // Global npm packages are typically in: /usr/local/lib/node_modules/must-cdk/
  // or ~/.npm-global/lib/node_modules/must-cdk/
  const possibleGlobalPaths = [
    path.resolve(currentDir, "../../src/amplify/example"), // lib/cli/commands -> src/amplify/example
    path.resolve(currentDir, "../../../src/amplify/example"), // if nested deeper
  ];

  for (const possiblePath of possibleGlobalPaths) {
    if (fs.existsSync(possiblePath)) {
      return possiblePath;
    }
  }

  // Last resort: try relative to process.cwd() (shouldn't normally be needed)
  examplePath = path.resolve(process.cwd(), "src/amplify/example");

  return examplePath;
}
