import { Duration } from "aws-cdk-lib";
import * as acm from "aws-cdk-lib/aws-certificatemanager";
import * as cloudfront from "aws-cdk-lib/aws-cloudfront";
import * as origins from "aws-cdk-lib/aws-cloudfront-origins";
import * as route53 from "aws-cdk-lib/aws-route53";
import * as targets from "aws-cdk-lib/aws-route53-targets";
import * as s3 from "aws-cdk-lib/aws-s3";
import { Construct } from "constructs";
import {
  CloudFrontToOriginsProps,
  S3OriginConfig,
  HttpOriginConfig,
  S3OriginInfo,
  HttpOriginInfo,
} from "./interface";

export class CloudFrontToOrigins extends Construct {
  public readonly distribution: cloudfront.Distribution;
  public readonly logBucket?: s3.IBucket;
  public readonly certificate?: acm.ICertificate;
  public readonly aRecords: route53.ARecord[] = [];
  public readonly domainNames?: string[];

  private readonly s3BucketsMap: Map<string, s3.IBucket> = new Map();
  private readonly httpOriginsMap: Map<string, origins.HttpOrigin> = new Map();
  private readonly s3OriginsMap: Map<string, cloudfront.IOrigin> = new Map();

  constructor(scope: Construct, id: string, props: CloudFrontToOriginsProps) {
    super(scope, id);

    // Validate that at least one origin is provided
    if (
      (!props.s3Origins || props.s3Origins.length === 0) &&
      (!props.httpOrigins || props.httpOrigins.length === 0)
    ) {
      throw new Error(
        "At least one origin (an S3 Bucket or a HTTP domain) must be provided",
      );
    }

    // Apply intelligent defaults
    const enhancedProps = this.applyDefaults(props);

    let certificate: acm.ICertificate | undefined;
    let logBucket: s3.IBucket | undefined;

    // Setup all S3 origins
    if (enhancedProps.s3Origins) {
      for (const s3Config of enhancedProps.s3Origins) {
        this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
        const s3Origin = this.setupS3Origin(s3Config);
        this.s3OriginsMap.set(s3Config.id, s3Origin);
      }
    }

    // Setup all HTTP origins
    if (enhancedProps.httpOrigins) {
      for (const httpConfig of enhancedProps.httpOrigins) {
        const httpOrigin = this.setupHttpOrigin(httpConfig);
        this.httpOriginsMap.set(httpConfig.id, httpOrigin);
      }
    }

    // Setup custom domain and certificate
    if (enhancedProps.customDomainName) {
      certificate = this.setupCustomDomain(enhancedProps);
    }

    // Setup logging (enabled by default unless explicitly disabled)
    if (enhancedProps.enableLogging !== false) {
      logBucket = this.setupLogging(enhancedProps);
    }

    // Create CloudFront distribution
    this.distribution = this.createDistribution(enhancedProps, {
      certificate,
      logBucket,
    });

    // Setup Route53 records for all domains
    if (
      enhancedProps.createRoute53Records !== false &&
      enhancedProps.hostedZone
    ) {
      this.setupRoute53Records(enhancedProps);
    }

    // Assign readonly properties
    this.logBucket = logBucket;
    this.certificate = certificate;
    this.domainNames = this.getAllDomainNames(enhancedProps);
  }

  /**
   * Get all domain names (primary + additional)
   */
  private getAllDomainNames(
    props: CloudFrontToOriginsProps,
  ): string[] | undefined {
    const domains: string[] = [];

    if (props.customDomainName) {
      domains.push(props.customDomainName);
    }

    if (props.additionalDomainNames) {
      domains.push(...props.additionalDomainNames);
    }

    return domains.length > 0 ? domains : undefined;
  }

  /**
   * Apply intelligent defaults based on origin configuration
   */
  private applyDefaults(
    props: CloudFrontToOriginsProps,
  ): CloudFrontToOriginsProps {
    return {
      // Apply base defaults
      enableLogging: true,
      createRoute53Records: true,
      priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
      enabled: true,
      // Override with user props
      ...props,
      // Apply HTTP origin protocol defaults
      httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
        protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
        ...httpOrigin,
      })),
      // Merge cache behaviors
      cacheBehaviors: [...(props.cacheBehaviors || [])],
    };
  }

  /**
   * Setup S3 origin using Origin Access Identity (OAI)
   */
  private setupS3Origin(s3Config: S3OriginConfig): cloudfront.IOrigin {
    // Use Origin Access Identity
    const originAccessIdentity =
      s3Config.originAccessIdentity ||
      new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
        comment: `OAI for ${s3Config.bucket.bucketName}`,
      });

    // Grant CloudFront access to S3 bucket
    s3Config.bucket.grantRead(originAccessIdentity);

    return origins.S3BucketOrigin.withOriginAccessIdentity(s3Config.bucket, {
      originAccessIdentity,
      originPath: s3Config.originPath,
      originId: s3Config.id,
      ...s3Config.s3OriginProps,
    });
  }

  /**
   * Setup HTTP origin
   */
  private setupHttpOrigin(httpConfig: HttpOriginConfig): origins.HttpOrigin {
    return new origins.HttpOrigin(httpConfig.domainName, {
      originPath: httpConfig.originPath,
      protocolPolicy:
        httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
      httpPort: httpConfig.httpPort,
      httpsPort: httpConfig.httpsPort,
      originId: httpConfig.id,
      ...httpConfig.httpOriginProps,
    });
  }

  /**
   * Setup custom domain and certificate
   */
  private setupCustomDomain(props: CloudFrontToOriginsProps): acm.ICertificate {
    // If certificate is provided directly, use it
    if (props.certificate) {
      return props.certificate;
    }

    // Auto-create certificate in us-east-1 if hostedZone is provided
    if (props.hostedZone && props.customDomainName) {
      return this.createCrossRegionCertificate(props);
    }

    throw new Error(
      "Either certificate or hostedZone must be provided for custom domain. " +
        "If hostedZone is provided, certificate will be automatically created in us-east-1.",
    );
  }

  /**
   * Create certificate in us-east-1 for CloudFront
   */
  private createCrossRegionCertificate(
    props: CloudFrontToOriginsProps,
  ): acm.ICertificate {
    const allDomains = this.getAllDomainNames(props) || [
      props.customDomainName!,
    ];

    // Use DnsValidatedCertificate to create certificate in us-east-1
    return new acm.DnsValidatedCertificate(this, "Certificate", {
      domainName: props.customDomainName!,
      subjectAlternativeNames:
        allDomains.length > 1 ? allDomains.slice(1) : undefined,
      hostedZone: props.hostedZone!,
      region: "us-east-1",
    });
  }

  /**
   * Setup CloudFront logging
   */
  private setupLogging(props: CloudFrontToOriginsProps): s3.IBucket {
    return (
      props.logBucket ||
      new s3.Bucket(this, "LogBucket", {
        blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        versioned: false,
        lifecycleRules: [
          {
            expiration: Duration.days(90),
            transitions: [
              {
                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: Duration.days(30),
              },
            ],
          },
        ],
      })
    );
  }

  /**
   * Get origin by ID
   */
  private getOriginById(originId: string): cloudfront.IOrigin {
    // Check S3 origins first
    const s3Origin = this.s3OriginsMap.get(originId);
    if (s3Origin) {
      return s3Origin;
    }

    // Check HTTP origins
    const httpOrigin = this.httpOriginsMap.get(originId);
    if (httpOrigin) {
      return httpOrigin;
    }

    throw new Error(`Origin with ID '${originId}' not found`);
  }

  /**
   * Determine default origin ID
   */
  private getDefaultOriginId(props: CloudFrontToOriginsProps): string {
    if (props.defaultOriginId) {
      return props.defaultOriginId;
    }

    // Default to first HTTP origin, then first S3 origin
    if (props.httpOrigins && props.httpOrigins.length > 0) {
      return props.httpOrigins[0].id;
    }

    if (props.s3Origins && props.s3Origins.length > 0) {
      return props.s3Origins[0].id;
    }

    throw new Error("No origins configured");
  }

  /**
   * Create CloudFront distribution
   */
  private createDistribution(
    props: CloudFrontToOriginsProps,
    resources: {
      certificate?: acm.ICertificate;
      logBucket?: s3.IBucket;
    },
  ): cloudfront.Distribution {
    // Determine default origin
    const defaultOriginId = this.getDefaultOriginId(props);
    const defaultOrigin = this.getOriginById(defaultOriginId);

    const defaultBehavior: cloudfront.BehaviorOptions = {
      origin: defaultOrigin,
      viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
      compress: true,
    };

    // Build additional behaviors
    const additionalBehaviors: Record<string, cloudfront.BehaviorOptions> = {};
    if (props.cacheBehaviors) {
      for (const behavior of props.cacheBehaviors) {
        const origin = this.getOriginById(behavior.originId);

        // Build behavior options object properly
        const behaviorConfig: any = {
          origin,
          viewerProtocolPolicy:
            behavior.viewerProtocolPolicy ||
            cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
          allowedMethods: behavior.allowedMethods,
          cachedMethods: behavior.cachedMethods,
          compress: behavior.compress ?? true,
        };

        // Add cache policy (prioritize direct policy over ID)
        if (behavior.cachePolicy) {
          behaviorConfig.cachePolicy = behavior.cachePolicy;
        } else if (behavior.cachePolicyId) {
          behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(
            this,
            `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`,
            behavior.cachePolicyId,
          );
        }

        // Add origin request policy
        if (behavior.originRequestPolicy) {
          behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
        } else if (behavior.originRequestPolicyId) {
          behaviorConfig.originRequestPolicy =
            cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(
              this,
              `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`,
              behavior.originRequestPolicyId,
            );
        }

        // Add response headers policy
        if (behavior.responseHeadersPolicy) {
          behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
        } else if (behavior.responseHeadersPolicyId) {
          behaviorConfig.responseHeadersPolicy =
            cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(
              this,
              `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`,
              behavior.responseHeadersPolicyId,
            );
        }

        additionalBehaviors[behavior.pathPattern] =
          behaviorConfig as cloudfront.BehaviorOptions;
      }
    }

    // Ensure all origins are referenced by creating behaviors for unused origins
    const usedOriginIds = new Set([defaultOriginId]);

    // Track origins used in cache behaviors
    if (props.cacheBehaviors) {
      props.cacheBehaviors.forEach((behavior) => {
        usedOriginIds.add(behavior.originId);
      });
    }

    // Create default behaviors for unused origins
    const allOriginIds = [
      ...this.s3OriginsMap.keys(),
      ...this.httpOriginsMap.keys(),
    ];
    allOriginIds.forEach((originId) => {
      if (!usedOriginIds.has(originId)) {
        const origin = this.getOriginById(originId);
        // Create a unique path pattern for this origin (lowercase)
        const pathPattern = `/${originId.toLowerCase()}/*`;
        additionalBehaviors[pathPattern] = {
          origin,
          viewerProtocolPolicy:
            cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
          compress: true,
        };
      }
    });

    // Prepare distribution props
    const distributionProps: cloudfront.DistributionProps = {
      comment: props.comment || `CloudFront distribution for ${this.node.id}`,
      defaultBehavior,
      additionalBehaviors,
      domainNames: this.getAllDomainNames(props),
      certificate: resources.certificate,
      defaultRootObject: props.defaultRootObject,
      enabled: props.enabled,
      priceClass: props.priceClass,
      geoRestriction: props.geoRestriction,
      enableLogging: props.enableLogging !== false,
      logBucket: resources.logBucket,
      logFilePrefix: props.logPrefix,
      logIncludesCookies: props.logIncludeCookies,
      errorResponses: props.errorPages,
      webAclId: props.webAclId,
      enableIpv6: props.enableIpv6,
      httpVersion: props.httpVersion,
    };

    return new cloudfront.Distribution(this, "Distribution", distributionProps);
  }

  /**
   * Setup Route53 alias records for all domains
   */
  private setupRoute53Records(props: CloudFrontToOriginsProps): void {
    if (!props.hostedZone) {
      return;
    }

    const domains = this.getAllDomainNames(props);
    if (!domains) {
      return;
    }

    // Create A record for each domain
    domains.forEach((domain, index) => {
      const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;

      const aRecord = new route53.ARecord(this, recordId, {
        zone: props.hostedZone!,
        recordName: domain,
        target: route53.RecordTarget.fromAlias(
          new targets.CloudFrontTarget(this.distribution),
        ),
      });

      this.aRecords.push(aRecord);
    });
  }

  /**
   * Get S3 bucket by origin ID
   */
  public getS3Bucket(originId: string): s3.IBucket | undefined {
    return this.s3BucketsMap.get(originId);
  }

  /**
   * Get HTTP origin by origin ID
   */
  public getHttpOrigin(originId: string): origins.HttpOrigin | undefined {
    return this.httpOriginsMap.get(originId);
  }

  /**
   * Get all S3 bucket origin IDs
   */
  public get s3OriginIds(): string[] {
    return Array.from(this.s3BucketsMap.keys());
  }

  /**
   * Get all HTTP origin IDs
   */
  public get httpOriginIds(): string[] {
    return Array.from(this.httpOriginsMap.keys());
  }

  /**
   * Get all S3 buckets as an array of objects with ID and bucket
   */
  public get s3Origins(): S3OriginInfo[] {
    return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
      id,
      bucket,
    }));
  }

  /**
   * Get all HTTP origins as an array of objects with ID and origin
   */
  public get httpOrigins(): HttpOriginInfo[] {
    return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
      id,
      origin,
    }));
  }

  /**
   * Get the CloudFront distribution domain name
   */
  public get distributionDomainName(): string {
    return this.distribution.distributionDomainName;
  }

  /**
   * Get the CloudFront distribution URL with protocol
   */
  public get distributionUrl(): string {
    return `https://${this.distribution.distributionDomainName}`;
  }

  /**
   * Get the custom domain URL (if configured)
   */
  public get customDomainUrl(): string | undefined {
    return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
  }
}
