import {
  Duration,
  aws_ec2 as ec2,
  aws_ecs as ecs,
  aws_elasticloadbalancingv2 as elbv2,
  aws_codedeploy as codedeploy,
  aws_logs as logs,
  aws_iam as iam,
  RemovalPolicy,
} from "aws-cdk-lib";
import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import { Construct } from "constructs";

/**
 * Configuration for the ECS Fargate task definition and container.
 */
export interface ContainerProps {
  /** Hard memory limit in MiB for the task (default: 2048). */
  readonly memoryLimit?: number;

  /** Soft memory reservation in MiB for the container (default: 1024). */
  readonly memoryReservation?: number;

  /** Container image to deploy. */
  readonly image: ecs.ContainerImage;

  /** Optional container health check configuration. */
  readonly healthCheck?: ecs.HealthCheck;

  /** The port number the container listens on. */
  readonly containerPort: number;
}

/**
 * Configuration for ECS service auto-scaling.
 */
export interface AutoScalingProps {
  /** Minimum number of tasks to run. */
  readonly minCapacity: number;

  /** Maximum number of tasks to run. */
  readonly maxCapacity: number;

  /** Scale task based on CPU utilization. */
  readonly cpuScale?: ecs.CpuUtilizationScalingProps;

  /** Scale task based on memory utilization. */
  readonly memoryScale?: ecs.MemoryUtilizationScalingProps;
}

/**
 * Properties for the EcsCodeDeploy construct.
 */
export interface EcsCodeDeployProps {
  /** VPC in which to deploy ECS and ALB resources. */
  readonly vpc: ec2.IVpc;

  /** Security group config */
  readonly securityGroups: ec2.ISecurityGroup[];

  /** Select which subnets the Service and ALB will placed on  */
  readonly subnets: ec2.SubnetSelection;

  /** ECS Cluster where the service will run. */
  readonly cluster: ecs.ICluster;

  /** Task role for the ECS task */
  readonly taskRole?: iam.IRole;

  /** Task execution role for the ECS task */
  readonly taskExecRole?: iam.IRole;

  /** Base name used for resources like log groups, roles, services, etc. */
  readonly serviceName: string;

  /** CPU units for the task (default: 1024). */
  readonly taskCPU?: number;

  readonly memoryLimit?: number;

  /** Configuration related to the task definition and container. */
  readonly containers: ContainerProps[];

  /** The ALB target port */
  readonly albTargetPort?: number;

  /** Whether the load balancer should be internet-facing (default: false). */
  readonly enablePublicLoadBalancer?: boolean;

  /** Optional ACM certificates for HTTPS termination. */
  readonly certificates: ICertificate[];

  /** Optional auto-scaling configuration. */
  readonly autoScaling?: AutoScalingProps;
}

export class EcsCodeDeploy extends Construct {
  public readonly service: ecs.FargateService;
  public readonly loadBalancer: elbv2.ApplicationLoadBalancer;
  public readonly taskDef: ecs.TaskDefinition;
  public readonly blueTargetGroup: elbv2.ApplicationTargetGroup;
  public readonly greenTargetGroup: elbv2.ApplicationTargetGroup;
  public readonly codeDeployApp: codedeploy.EcsApplication;

  public readonly taskRole: iam.IRole;
  public readonly taskExecutionRole: iam.IRole;

  private readonly props: EcsCodeDeployProps;
  private readonly logGroup: logs.LogGroup;

  constructor(scope: Construct, id: string, props: EcsCodeDeployProps) {
    super(scope, id);
    this.props = props;

    this.logGroup = this.createLogGroup();

    const { taskRole, taskExecRole } = this.createRole();

    this.taskRole = taskRole;
    this.taskExecutionRole = taskExecRole;

    this.taskDef = this.createTaskDefinition();
    this.service = this.createEcsService();

    this.loadBalancer = this.createApplicationLoadBalancer();
    const { blueTargetGroup, greenTargetGroup } = this.createTargetGroups();
    this.blueTargetGroup = blueTargetGroup;
    this.greenTargetGroup = greenTargetGroup;

    this.setupAutoScaling();

    this.codeDeployApp = this.createCodeDeployApplication();
    this.createCodeDeployDeploymentGroup();
  }

  private createRole(): { taskRole: iam.IRole; taskExecRole: iam.IRole } {
    const taskRole =
      this.props.taskRole ??
      new iam.Role(this, "TaskRole", {
        roleName: `${this.props.serviceName}-task-role`,
        assumedBy: new iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
        description: `Task role for ${this.props.serviceName}`,
      });

    const taskExecRole =
      this.props.taskExecRole ??
      new iam.Role(this, "TaskExecRole", {
        roleName: `${this.props.serviceName}-task-exec-role`,
        assumedBy: new iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
        description: `Task execution role for ${this.props.serviceName}`,
        managedPolicies: [
          iam.ManagedPolicy.fromAwsManagedPolicyName(
            "service-role/AmazonECSTaskExecutionRolePolicy",
          ),
        ],
      });

    return { taskRole, taskExecRole };
  }

  private createLogGroup(): logs.LogGroup {
    const { serviceName } = this.props;

    return new logs.LogGroup(this, "LogGroup", {
      logGroupName: `/aws/ecs/${serviceName}`,
      retention: logs.RetentionDays.ONE_MONTH,
      removalPolicy: RemovalPolicy.DESTROY,
    });
  }

  private createTaskDefinition(): ecs.FargateTaskDefinition {
    const { taskCPU = 1024, memoryLimit = 2048, containers } = this.props;

    const taskDef = new ecs.FargateTaskDefinition(this, "TaskDefinition", {
      memoryLimitMiB: memoryLimit,
      cpu: taskCPU,
      executionRole: this.taskExecutionRole,
      taskRole: this.taskRole,
    });

    // Add containers to task definition
    this.addContainersToTaskDefinition(taskDef, containers);

    return taskDef;
  }

  private addContainersToTaskDefinition(
    taskDef: ecs.FargateTaskDefinition,
    containers: ContainerProps[],
  ): void {
    containers.forEach((container, index) => {
      taskDef.addContainer(`Container${index}`, {
        image: container.image,
        memoryReservationMiB: container.memoryLimit ?? 1024,
        logging: ecs.LogDriver.awsLogs({
          streamPrefix: "ecs",
          logGroup: this.logGroup,
        }),
        portMappings: [{ containerPort: container.containerPort }],
        healthCheck: container.healthCheck ?? {
          command: [
            "CMD-SHELL",
            `curl -f http://localhost:${container.containerPort} || exit 1`,
          ],
          interval: Duration.seconds(30),
          timeout: Duration.seconds(5),
        },
      });
    });
  }

  private createEcsService(): ecs.FargateService {
    const { cluster, serviceName, subnets, securityGroups } = this.props;

    return new ecs.FargateService(this, "Service", {
      cluster,
      serviceName: `${serviceName}-service`,
      taskDefinition: this.taskDef,
      assignPublicIp: true,
      desiredCount: 1,
      deploymentController: {
        type: ecs.DeploymentControllerType.CODE_DEPLOY,
      },
      healthCheckGracePeriod: Duration.seconds(300),
      vpcSubnets: subnets,
      securityGroups: securityGroups,
    });
  }

  private createApplicationLoadBalancer(): elbv2.ApplicationLoadBalancer {
    const { vpc, subnets, enablePublicLoadBalancer = false } = this.props;

    const alb = new elbv2.ApplicationLoadBalancer(this, "LoadBalancer", {
      vpc,
      loadBalancerName: `${this.props.serviceName}-alb`,
      internetFacing: enablePublicLoadBalancer,
      vpcSubnets: subnets,
    });

    // Add HTTP to HTTPS redirect
    alb.addRedirect();

    return alb;
  }

  private createTargetGroups(): {
    blueTargetGroup: elbv2.ApplicationTargetGroup;
    greenTargetGroup: elbv2.ApplicationTargetGroup;
  } {
    const { certificates, containers } = this.props;
    const primaryContainer = containers[0];
    const targetPort =
      this.props.albTargetPort ?? primaryContainer.containerPort;

    // Create HTTPS listeners
    const httpsBlueListener = this.loadBalancer.addListener(
      "BlueHttpsListener",
      {
        port: 443,
        protocol: elbv2.ApplicationProtocol.HTTPS,
        certificates: certificates,
      },
    );

    const httpsGreenListener = this.loadBalancer.addListener(
      "GreenHttpsListener",
      {
        port: 8080,
        protocol: elbv2.ApplicationProtocol.HTTPS,
        certificates: certificates,
      },
    );

    // Health check configuration
    const healthCheck: elbv2.HealthCheck = {
      path: "/",
      interval: Duration.seconds(30),
      timeout: Duration.seconds(10),
      healthyThresholdCount: 2,
      unhealthyThresholdCount: 5,
    };

    // Create target groups
    const blueTargetGroup = httpsBlueListener.addTargets("BlueTarget", {
      port: targetPort,
      protocol: elbv2.ApplicationProtocol.HTTP,
      targets: [this.service],
      healthCheck: healthCheck,
      stickinessCookieDuration: Duration.days(1),
    });

    const greenTargetGroup = httpsGreenListener.addTargets("GreenTarget", {
      port: targetPort,
      protocol: elbv2.ApplicationProtocol.HTTP,
      targets: [], // Empty for green deployment
      healthCheck: healthCheck,
      stickinessCookieDuration: Duration.days(1),
    });

    return { blueTargetGroup, greenTargetGroup };
  }

  private setupAutoScaling(): void {
    const { autoScaling } = this.props;

    if (!autoScaling) {
      return;
    }

    const scalableTarget = this.service.autoScaleTaskCount({
      minCapacity: autoScaling.minCapacity,
      maxCapacity: autoScaling.maxCapacity,
    });

    // Setup CPU-based scaling if configured
    if (autoScaling.cpuScale) {
      scalableTarget.scaleOnCpuUtilization("CpuScaling", autoScaling.cpuScale);
    }

    // Setup memory-based scaling if configured
    if (autoScaling.memoryScale) {
      scalableTarget.scaleOnMemoryUtilization(
        "MemoryScaling",
        autoScaling.memoryScale,
      );
    }
  }

  private createCodeDeployApplication(): codedeploy.EcsApplication {
    const { serviceName } = this.props;

    return new codedeploy.EcsApplication(this, "CodeDeployApp", {
      applicationName: serviceName,
    });
  }

  private createCodeDeployDeploymentGroup(): codedeploy.EcsDeploymentGroup {
    const { serviceName } = this.props;

    return new codedeploy.EcsDeploymentGroup(this, "DeploymentGroup", {
      deploymentGroupName: serviceName,
      application: this.codeDeployApp,
      service: this.service,
      blueGreenDeploymentConfig: {
        blueTargetGroup: this.blueTargetGroup,
        greenTargetGroup: this.greenTargetGroup,
        listener: this.loadBalancer.listeners[0],
      },
    });
  }

  public blueListener(): elbv2.ApplicationListener {
    return (
      this.loadBalancer.listeners.find((listener) =>
        listener.listenerArn.includes("BlueHttpsListener"),
      ) || this.loadBalancer.listeners[0]
    );
  }

  public greenListener(): elbv2.ApplicationListener {
    return (
      this.loadBalancer.listeners.find((listener) =>
        listener.listenerArn.includes("GreenHttpsListener"),
      ) || this.loadBalancer.listeners[1]
    );
  }

  public allListeners(): elbv2.ApplicationListener[] {
    return this.loadBalancer.listeners;
  }

  public loadBalancerDnsName(): string {
    return this.loadBalancer.loadBalancerDnsName;
  }

  public serviceArn(): string {
    return this.service.serviceArn;
  }
}
