# Copyright 2018-2024 Jérôme Dumonteil
# Copyright (c) 2009-2010 Ars Aperta, Itaapy, Pierlis, Talend.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Authors (odfdo project): jerome.dumonteil@gmail.com
# The odfdo project is a derivative work of the lpod-python project:
# https://github.com/lpod/lpod-python
# Authors: Hervé Cauwelier <herve@itaapy.com>
#          Romain Gauthier <romain@itaapy.com>
#          Jerome Dumonteil <jerome.dumonteil@itaapy.com>
"""Note class for "text:note" and Annotation class for "office:annotation".
"""
from __future__ import annotations

from datetime import datetime
from typing import Any

from .element import Element, PropDef, register_element_class
from .mixin_dc_creator import DcCreatorMixin
from .mixin_dc_date import DcDateMixin


def get_unique_office_name(element: Element | None = None) -> str:
    """Provide an autogenerated unique "office:name" for the document."""
    if element is not None:
        body = element.document_body
    else:
        body = None
    if body:
        used = set(body.get_office_names())
    else:
        used = set()
    # unplugged current paragraph:
    if element is not None:
        used.update(element.get_office_names())
    indice = 1
    while True:
        name = f"__Fieldmark__lpod_{indice}"
        if name in used:
            indice += 1
            continue
        break
    return name


class Note(Element):
    """Either a footnote or a endnote element with the given text,
    optionally referencing it using the given note_id.

    Arguments:

        note_class -- 'footnote' or 'endnote'

        note_id -- str

        citation -- str

        body -- str or Element
    """

    _tag = "text:note"
    _properties = (
        PropDef("note_class", "text:note-class"),
        PropDef("note_id", "text:id"),
    )

    def __init__(
        self,
        note_class: str = "footnote",
        note_id: str | None = None,
        citation: str | None = None,
        body: str | None = None,
        **kwargs: Any,
    ) -> None:
        super().__init__(**kwargs)
        if self._do_init:
            self.insert(Element.from_tag("text:note-body"), position=0)
            self.insert(Element.from_tag("text:note-citation"), position=0)
            self.note_class = note_class
            if note_id is not None:
                self.note_id = note_id
            if citation is not None:
                self.citation = citation
            if body is not None:
                self.note_body = body

    @property
    def citation(self) -> str:
        note_citation = self.get_element("text:note-citation")
        if note_citation:
            return note_citation.text
        return ""

    @citation.setter
    def citation(self, text: str | None) -> None:
        note_citation = self.get_element("text:note-citation")
        if note_citation:
            note_citation.text = text  # type:ignore

    @property
    def note_body(self) -> str:
        note_body = self.get_element("text:note-body")
        if note_body:
            return note_body.text_content
        return ""

    @note_body.setter
    def note_body(self, text_or_element: Element | str | None) -> None:
        note_body = self.get_element("text:note-body")
        if not note_body:
            return None
        if text_or_element is None:
            note_body.text_content = ""
        elif isinstance(text_or_element, str):
            note_body.text_content = text_or_element
        elif isinstance(text_or_element, Element):
            note_body.clear()
            note_body.append(text_or_element)
        else:
            raise TypeError(f'Unexpected type for body: "{type(text_or_element)}"')

    def check_validity(self) -> None:
        if not self.note_class:
            raise ValueError('Note class must be "footnote" or "endnote"')
        if not self.note_id:
            raise ValueError("Note must have an id")
        if not self.citation:
            raise ValueError("Note must have a citation")
        if not self.note_body:
            pass


Note._define_attribut_property()


class Annotation(Element, DcCreatorMixin, DcDateMixin):
    """Annotation element credited to the given creator with the
    given text, optionally dated (current date by default).
    If name not provided and some parent is provided, the name is
    autogenerated.

    Arguments:

        text -- str or odf_element

        creator -- str

        date -- datetime

        name -- str

        parent -- Element
    """

    _tag = "office:annotation"
    _properties = (
        PropDef("name", "office:name"),
        PropDef("note_id", "text:id"),
    )

    def __init__(
        self,
        text_or_element: Element | str | None = None,
        creator: str | None = None,
        date: datetime | None = None,
        name: str | None = None,
        parent: Element | None = None,
        **kwargs: Any,
    ) -> None:
        # fixme : use offset
        # TODO allow paragraph and text styles
        super().__init__(**kwargs)

        if self._do_init:
            self.note_body = text_or_element  # type:ignore
            if creator:
                self.creator = creator
            if date is None:
                date = datetime.now()
            self.date = date
            if not name:
                name = get_unique_office_name(parent)
                self.name = name

    @property
    def dc_creator(self) -> str | None:
        """Alias for self.creator property."""
        return self.creator

    @dc_creator.setter
    def dc_creator(self, creator: str) -> None:
        self.creator = creator

    @property
    def dc_date(self) -> datetime | None:
        """Alias for self.date property."""
        return self.date

    @dc_date.setter
    def dc_date(self, dtdate: datetime) -> None:
        self.date = dtdate

    @property
    def note_body(self) -> str:
        return self.text_content

    @note_body.setter
    def note_body(self, text_or_element: Element | str | None) -> None:
        if text_or_element is None:
            self.text_content = ""
        elif isinstance(text_or_element, str):
            self.text_content = text_or_element
        elif isinstance(text_or_element, Element):
            self.clear()
            self.append(text_or_element)
        else:
            raise TypeError(f'Unexpected type for body: "{type(text_or_element)}"')

    @property
    def start(self) -> Element:
        """Return self."""
        return self

    @property
    def end(self) -> Element | None:
        """Return the corresponding annotation-end tag or None."""
        name = self.name
        parent = self.parent
        if parent is None:
            raise ValueError("Can't find end tag: no parent available")
        body = self.document_body
        if not body:
            body = parent
        return body.get_annotation_end(name=name)

    def get_annotated(
        self,
        as_text: bool = False,
        no_header: bool = True,
        clean: bool = True,
    ) -> Element | list | str | None:
        """Returns the annotated content from an annotation.

        If no content exists (single position annotation or annotation-end not
        found), returns [] (or "" if text flag is True).
        If as_text is True: returns the text content.
        If clean is True: suppress unwanted tags (deletions marks, ...)
        If no_header is True: existing text:h are changed in text:p
        By default: returns a list of odf_element, cleaned and without headers.

        Arguments:

            as_text -- boolean

            clean -- boolean

            no_header -- boolean

        Return: list or Element or text or None
        """
        end = self.end
        if end is None:
            if as_text:
                return ""
            return None
        body = self.document_body
        if not body:
            body = self.root
        return body.get_between(
            self, end, as_text=as_text, no_header=no_header, clean=clean
        )

    def delete(self, child: Element | None = None, keep_tail: bool = True) -> None:
        """Delete the given element from the XML tree. If no element is given,
        "self" is deleted. The XML library may allow to continue to use an
        element now "orphan" as long as you have a reference to it.

        For Annotation : delete the annotation-end tag if exists.

        Arguments:

            child -- Element or None
        """
        if child is not None:  # act like normal delete
            super().delete(child)
            return
        end = self.end
        if end:
            end.delete()
        # act like normal delete
        super().delete()

    def check_validity(self) -> None:
        if not self.note_body:
            raise ValueError("Annotation must have a body")
        if not self.dc_creator:
            raise ValueError("Annotation must have a creator")
        if not self.dc_date:
            self.dc_date = datetime.now()


Annotation._define_attribut_property()


class AnnotationEnd(Element):
    """AnnotationEnd: the "office:annotation-end" element may be used to
    define the end of a text range of document content that spans element
    boundaries. In that case, an "office:annotation" element shall precede
    the "office:annotation-end" element. Both elements shall have the same
    value for their office:name attribute. The "office:annotation-end" element
    shall be preceded by an "office:annotation" element that has the same
    value for its office:name attribute as the "office:annotation-end"
    element. An "office:annotation-end" element without a preceding
    "office:annotation" element that has the same name assigned is ignored.
    """

    _tag = "office:annotation-end"
    _properties = (PropDef("name", "office:name"),)

    def __init__(
        self,
        annotation: Element | None = None,
        name: str | None = None,
        **kwargs: Any,
    ) -> None:
        """Initialize an AnnotationEnd element. Either annotation or name must be
        provided to have proper reference for the annotation-end.

        Arguments:

            annotation -- odf_annotation element

            name -- str
        """
        # fixme : use offset
        # TODO allow paragraph and text styles
        super().__init__(**kwargs)
        if self._do_init:
            if annotation:
                name = annotation.name  # type: ignore
            if not name:
                raise ValueError("Annotation-end must have a name")
            self.name = name

    @property
    def start(self) -> Element | None:
        """Return the corresponding annotation starting tag or None."""
        name = self.name
        parent = self.parent
        if parent is None:
            raise ValueError("Can't find start tag: no parent available")
        body = self.document_body
        if not body:
            body = parent
        return body.get_annotation(name=name)

    @property
    def end(self) -> Element:
        """Return self."""
        return self


AnnotationEnd._define_attribut_property()

register_element_class(Note)
register_element_class(Annotation)
register_element_class(AnnotationEnd)
