"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = exports.StorageMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib/core");
const core_1 = require("aws-cdk-lib/core");
const cr = require("aws-cdk-lib/custom-resources");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The storage mode for the cluster brokers.
 */
var StorageMode;
(function (StorageMode) {
    /**
     * Local storage mode utilizes network attached EBS storage.
     */
    StorageMode["LOCAL"] = "LOCAL";
    /**
     * Tiered storage mode utilizes EBS storage and Tiered storage.
     */
    StorageMode["TIERED"] = "TIERED";
})(StorageMode || (exports.StorageMode = StorageMode = {}));
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption || (exports.ClientBrokerEncryption = ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * SASL authentication
     */
    static sasl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sasl);
            }
            throw error;
        }
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tls);
            }
            throw error;
        }
        return new ClientAuthentication(undefined, props);
    }
    /**
     * SASL + TLS authentication
     */
    static saslTls(saslTlsProps) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslTlsAuthProps(saslTlsProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.saslTls);
            }
            throw error;
        }
        return new ClientAuthentication(saslTlsProps, saslTlsProps);
    }
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.145.0-alpha.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (props.clientAuthentication?.saslProps?.iam &&
            props.clientAuthentication?.saslProps?.scram) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (props.clientAuthentication?.saslProps?.scram ||
                props.clientAuthentication?.saslProps?.iam)) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = props.ebsStorageInfo?.volumeSize ?? 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        if (props.storageMode && props.storageMode === StorageMode.TIERED) {
            if (!props.kafkaVersion.isTieredStorageCompatible()) {
                throw Error(`To deploy a tiered cluster you must select a compatible Kafka version, got ${props.kafkaVersion.version}`);
            }
            if (instanceType === this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.SMALL))) {
                throw Error('Tiered storage doesn\'t support broker type t3.small');
            }
        }
        const encryptionAtRest = props.ebsStorageInfo?.encryptionKey
            ? {
                dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
            }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: props.encryptionInTransit?.clientBroker ??
                ClientBrokerEncryption.TLS,
            inCluster: props.encryptionInTransit?.enableInCluster ?? true,
        };
        const openMonitoring = props.monitoring?.enablePrometheusJmxExporter ||
            props.monitoring?.enablePrometheusNodeExporter
            ? {
                prometheus: {
                    jmxExporter: props.monitoring?.enablePrometheusJmxExporter
                        ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: props.monitoring
                        ?.enablePrometheusNodeExporter
                        ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingBucket = props.logging?.s3?.bucket;
        if (loggingBucket && core_1.FeatureFlags.of(this).isEnabled(cx_api_1.S3_CREATE_DEFAULT_LOGGING_POLICY)) {
            const stack = core.Stack.of(this);
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [
                    loggingBucket.arnForObjects(`AWSLogs/${stack.account}/*`),
                ],
                actions: ['s3:PutObject'],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [loggingBucket.bucketArn],
                actions: [
                    's3:GetBucketAcl',
                    's3:ListBucket',
                ],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
        }
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: props.logging?.cloudwatchLogGroup !== undefined,
                    logGroup: props.logging?.cloudwatchLogGroup?.logGroupName,
                },
                firehose: {
                    enabled: props.logging?.firehoseDeliveryStreamName !==
                        undefined,
                    deliveryStream: props.logging?.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: loggingBucket !== undefined,
                    bucket: loggingBucket?.bucketName,
                    prefix: props.logging?.s3?.prefix,
                },
            },
        };
        if (props.clientAuthentication?.saslProps?.scram &&
            props.clientAuthentication?.saslProps?.key === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if (props.clientAuthentication?.saslProps?.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
            if (props.clientAuthentication?.tlsProps) {
                clientAuthentication = {
                    sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
                    tls: {
                        certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities?.map((ca) => ca.certificateAuthorityArn),
                    },
                };
            }
        }
        else if (props.clientAuthentication?.saslProps?.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (props.clientAuthentication?.tlsProps?.certificateAuthorities !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: props.monitoring?.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            storageMode: props.storageMode,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                    // Limit the output of describeCluster that is otherwise too large
                    outputPaths: [
                        'ClusterInfo.ZookeeperConnectString',
                        'ClusterInfo.ZookeeperConnectStringTls',
                    ],
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
                installLatestAwsSdk: false,
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
                // APIs are available in 2.1055.0
                installLatestAwsSdk: false,
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * Get the list of brokers that a SASL/IAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersSaslIam() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslIam');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${(0, uniqueid_1.addressOf)(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [this.saslScramAuthenticationKey?.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
                installLatestAwsSdk: false,
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.145.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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