# -*- coding: utf-8 -*-
#
# michael a.g. aïvázis <michael.aivazis@para-sim.com>
# (c) 1998-2023 all rights reserved


# access to the environment variables
import os

# pull the control sequence generator
from .CSI import CSI


# maps of color names from a variety of color spaces to the ANSI control sequences required tp
# render them in compatible terminal emulators
class ANSI:
    """
    Encapsulation of mappings from color names to ANSI control sequences for a variety of color
    spaces
    """

    # interface
    @classmethod
    def compatible(cls):
        """
        Attempt to assess whether the current terminal is ANSI compatible
        """
        # get the compatible terminal types
        compatible = cls._compatible
        # try to read the value of the {TERM} environment variable
        term = os.environ.get("TERM", "dumb").lower()
        # check whether terminal type is compatible
        return term in compatible

    # color tables
    @classmethod
    def null(cls, name):
        """
        The null colorspace where every color maps to an empty string
        """
        # everything is an empty string
        return ""

    @classmethod
    def ansi(cls, name):
        """
        Return the ANSI control sequence required to render the given color {name}
        """
        # look it up and return it
        return cls._ansi.get(name, "")

    @classmethod
    def gray(cls, name):
        """
        Return the ANSI control sequence required to render the given color {name}
        """
        # look it up and return it
        return cls._gray.get(name, "")

    @classmethod
    def misc(cls, name):
        """
        Return the ANSI control sequence required to render the given color {name}
        """
        # look it up and return it
        return cls._misc.get(name, "")

    @classmethod
    def x11(cls, name):
        """
        Return the ANSI control sequence required to render the given color {name}
        """
        # look it up and return it
        return cls._x11.get(name, "")

    # implementation details
    _ansi = {
        # the reset sequence
        "normal": CSI.reset(),
        # regular colors
        "black": CSI.csi3(code=30),
        "red": CSI.csi3(code=31),
        "green": CSI.csi3(code=32),
        "brown": CSI.csi3(code=33),
        "blue": CSI.csi3(code=34),
        "purple": CSI.csi3(code=35),
        "cyan": CSI.csi3(code=36),
        "light-gray": CSI.csi3(code=37),
        # bright colors
        "dark-gray": CSI.csi3(code=30, bright=True),
        "light-red": CSI.csi3(code=31, bright=True),
        "light-green": CSI.csi3(code=32, bright=True),
        "yellow": CSI.csi3(code=33, bright=True),
        "light-blue": CSI.csi3(code=34, bright=True),
        "light-purple": CSI.csi3(code=35, bright=True),
        "light-cyan": CSI.csi3(code=36, bright=True),
        "white": CSI.csi3(code=37, bright=True),
    }

    _gray = {
        # the reset sequence
        "normal": CSI.reset(),
        # grays
        "gray10": CSI.csi24(0x19, 0x19, 0x19),
        "gray30": CSI.csi24(0x4C, 0x4C, 0x4C),
        "gray41": CSI.csi24(0x69, 0x69, 0x69),
        "gray50": CSI.csi24(0x80, 0x80, 0x80),
        "gray66": CSI.csi24(0xA9, 0xA9, 0xA9),
        "gray75": CSI.csi24(0xBE, 0xBE, 0xBE),
    }

    _misc = {
        # the reset sequence
        "normal": CSI.reset(),
        # other custom colors
        "amber": CSI.csi24(0xFF, 0xBF, 0x00),
        "sage": CSI.csi24(176, 208, 176),
    }

    # the {x11} table gets autogenerated from the canonical {rgb.txt} file
    from .ANSI_x11 import table as _x11

    # the list of compatible terminal types
    _compatible = {
        "ansi",
        "vt102",
        "vt220",
        "vt320",
        "vt420",
        "xterm",
        "xterm-color",
        "xterm-16color",
        "xterm-256color",
    }


# end of file
