#  Copyright (c) 2022. Harvard University
#
#  Developed by Research Software Engineering,
#  Faculty of Arts and Sciences, Research Computing (FAS RC)
#  Author: Michael A Bouzinier
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
"""
This loader is used to ingest files generated by SAS software
in SAS 7BDAT format. It looks for SAS 7BDAT files in a given path
matching a given pattern and loads the data into the database.

Before running the loader, one should generate database
schema by introspecting the metadata defined in the SAS 7BDAT
files.

It is used to ingest Medicare data for 1999 to 2010 years.

See also:
- :class:`.SASIntrospector` for introspection and schema analysis of SAS files.
"""

import copy
import os

from dorieh.platform.loader.data_loader import DataLoader

from dorieh.cms.tools.mcr_sas import MedicareSAS
from dorieh.platform.loader import LoaderConfig


class SASLoader(MedicareSAS):
    """
    Loader class that looks for SAS 7BDAT files in a given path
    matching a given pattern and loads the data in the database.

    This loader is used to ingest Medicare data for 1999 - 2010 years
    """

    @classmethod
    def process(cls):
        loader = SASLoader()
        loader.traverse(loader.pattern)

    def __init__(self):
        self.pattern = "[1-2]*/*/*.sas7bdat"
        self.context = LoaderConfig(__doc__)
        self.context.domain = "cms"
        self.domain = self.context.domain
        self.context.set_empty_args()
        super().__init__(
            root_dir=self.context.data
        )
        if not self.context.incremental and not self.context.sloppy:
            self.context.reset = True
        return

    def handle(self, table: str, file_path: str, file_type: str, year: int):
        """
        Handle a single SAS 7BDAT file by creating a data loader instance
        and invoking the loading process.

        This method overrides the abstract method from the parent class
        :class:`.MedicareSAS`.

        For each input file, it creates an instance of
        :class:`~dorieh.platform.loader.data_loader.DataLoader`
        configured with the appropriate context, and then runs it to ingest data.

        Args:
            table (str): Target table name for the data.
            file_path (str): Absolute path to the source SAS file.
            file_type (str): Type or category of the SAS file.
            year (int): Year associated with the data file.

        """

        context = copy.deepcopy(self.context)
        context.table = table
        context.pattern = [os.path.join("**", os.path.basename(file_path))]
        loader = DataLoader(context)
        loader.run()


if __name__ == '__main__':
    SASLoader.process()


