# Built-in Real-Time Message Display Feature

## Overview

Real-time message display is now a built-in feature of the synqed API! Users can enable colored, formatted message display with a single parameter instead of writing custom callback functions.

## What Changed

### Before (Old Approach)
```python
def message_callback(msg_dict):
    """Custom callback for displaying messages"""
    sender = msg_dict.get('sender_name', 'System')
    content = msg_dict.get('content', '')
    
    # Color code by agent
    if 'Research' in sender:
        icon = "🔍"
        color = "\033[92m"  # Green
    elif 'Writer' in sender:
        icon = "📝"
        color = "\033[93m"  # Yellow
    else:
        icon = "⚙️"
        color = "\033[90m"  # Gray
    
    reset = "\033[0m"
    display = content[:200] + "..." if len(content) > 200 else content
    
    print(f"{color}{icon} {sender}:{reset}")
    for line in display.split('\n'):
        if line.strip():
            print(f"  {line}")
    print()

workspace = synqed.OrchestratedWorkspace(
    orchestrator=orchestrator,
    enable_agent_discussion=True,
    message_callback=message_callback  # Custom callback required
)
```

### After (New Approach) ✨
```python
workspace = synqed.OrchestratedWorkspace(
    orchestrator=orchestrator,
    enable_agent_discussion=True,
    display_messages=True,  # That's it!
)
```

## Usage

### 1. WorkspaceEngine (via create_workspace)

```python
import synqed

# Simple usage
workspace = synqed.create_workspace(
    name="My Workspace",
    task="Collaborate on solving a problem",
    agents=[agent1, agent2, agent3],
    display_messages=True,  # Enable colored display
)

result = await workspace.run()
```

#### Verbose Mode

```python
workspace = synqed.create_workspace(
    name="My Workspace",
    task="Collaborate on solving a problem",
    agents=[agent1, agent2, agent3],
    display_messages=True,
    display_verbose=True,  # Show message type, phase, etc.
)
```

### 2. OrchestratedWorkspace

```python
import synqed

workspace = synqed.OrchestratedWorkspace(
    orchestrator=orchestrator,
    enable_agent_discussion=True,
    display_messages=True,  # Enable colored display
)

workspace.register_agent(agent1)
workspace.register_agent(agent2)

result = await workspace.execute_task("My complex task")
```

## Features

### Automatic Color Assignment
- Each agent is automatically assigned a distinct color
- Colors are consistent throughout the workspace session
- 11 distinct colors available for agents
- System messages shown in gray

### Clean Formatting
- Agent names are color-coded
- Messages are cleanly formatted
- Long messages are handled gracefully
- Works seamlessly with both WorkspaceEngine and OrchestratedWorkspace

### Backward Compatible
- Existing code continues to work
- Custom callbacks can still be used
- Built-in display can be combined with custom callbacks

## Combining with Custom Callbacks

The built-in display works alongside custom callbacks:

```python
def my_custom_callback(msg):
    # Your custom logic (logging, monitoring, etc.)
    logger.info(f"{msg.sender_name}: {msg.content}")

workspace = synqed.create_workspace(
    name="My Workspace",
    task="Collaborate on a task",
    agents=[agent1, agent2, agent3],
    display_messages=True,     # Built-in colored display
    on_message=my_custom_callback,  # Your custom logic
)
```

Both the built-in display AND your custom callback will be called!

## API Reference

### Parameters

#### `display_messages: bool = False`
Enable built-in colored message display for real-time agent communication.

#### `display_verbose: bool = False`
Show full message details including message type, phase, and other metadata. Only works when `display_messages=True`.

### Availability

The `display_messages` and `display_verbose` parameters are available in:
- `synqed.create_workspace()` - Primary API for creating workspaces
- `synqed.OrchestratedWorkspace()` - Advanced orchestration
- `synqed.WorkspaceConfig` - Low-level configuration

## Implementation Details

### Core Components

1. **ColoredAgentDisplay** (`synqed.display`)
   - Manages color assignment
   - Provides formatted output methods
   - Supports verbose mode

2. **WorkspaceEngine** Integration
   - Automatically creates ColoredAgentDisplay instance when `display_messages=True`
   - Chains callbacks if both display and custom callback are provided

3. **OrchestratedWorkspace** Integration
   - Wraps message dictionaries for display
   - Maintains color consistency across workspace operations

### Files Modified

1. `src/synqed/workspace_engine.py`
   - Added `display_messages` and `display_verbose` to `WorkspaceConfig`
   - Integrated `ColoredAgentDisplay` into `WorkspaceEngine.__init__()`
   - Automatic callback chaining

2. `src/synqed/builders.py`
   - Added `display_messages` and `display_verbose` parameters to `create_workspace()`
   - Updated documentation with examples

3. `src/synqed/orchestrated_workspace.py`
   - Added `display_messages` and `display_verbose` parameters to `__init__()`
   - Implemented display callback for message dictionaries
   - Updated documentation

4. `examples/intro/level3_two_agents.py`
   - Updated to use new built-in display feature
   - Removed custom callback code

5. `examples/intro/display_demo.py`
   - New demo showing all display features
   - Examples of simple and advanced usage

## Migration Guide

### From Custom Callbacks to Built-in Display

**Old Code:**
```python
def message_callback(msg_dict):
    # Your display logic...
    pass

workspace = synqed.OrchestratedWorkspace(
    orchestrator=orchestrator,
    message_callback=message_callback
)
```

**New Code:**
```python
workspace = synqed.OrchestratedWorkspace(
    orchestrator=orchestrator,
    display_messages=True
)
```

### Keeping Custom Logic

If you need to keep custom logic (logging, monitoring, etc.), just add it as an `on_message` callback:

```python
def my_logic(msg):
    # Your custom logic
    logger.info(f"Message from {msg.sender_name}")

workspace = synqed.create_workspace(
    name="My Workspace",
    agents=[...],
    display_messages=True,  # Built-in display
    on_message=my_logic     # Your logic
)
```

## Benefits

1. **Simplicity**: Enable display with one parameter
2. **Consistency**: All workspaces use the same display format
3. **Maintainability**: No custom display code to maintain
4. **Flexibility**: Can still add custom callbacks when needed
5. **Beauty**: Automatic color coding makes multi-agent systems easier to follow

## Examples

See the following examples for complete demonstrations:
- `examples/intro/display_demo.py` - Feature overview and usage patterns
- `examples/intro/level3_two_agents.py` - Updated to use built-in display

## Future Enhancements

Possible future additions:
- Emoji customization per agent
- Custom color schemes
- Output to file/stream
- HTML/Rich text output
- Web-based real-time dashboard

---

**Made with ❤️ for the Synqed community**

