"""
Display Demo - Built-in Real-Time Message Display

This example demonstrates the new built-in colored message display feature.
Instead of writing custom callbacks, just set display_messages=True!

Before (old way):
    def message_callback(msg_dict):
        sender = msg_dict.get('sender_name', 'System')
        content = msg_dict.get('content', '')
        # ... custom color logic ...
        print(f"{color}{sender}: {content}")
    
    workspace = synqed.OrchestratedWorkspace(
        orchestrator=orchestrator,
        message_callback=message_callback
    )

After (new way):
    workspace = synqed.OrchestratedWorkspace(
        orchestrator=orchestrator,
        display_messages=True  # That's it!
    )

The new approach:
- Automatically assigns distinct colors to each agent
- Formats messages beautifully
- No custom code needed
- Still allows custom callbacks if you want to do additional processing
"""

import asyncio
import os
from pathlib import Path
from dotenv import load_dotenv
import synqed

# Load environment
load_dotenv()
load_dotenv(dotenv_path=Path(__file__).parent / ".env")


async def main():
    print("=" * 80)
    print("  SYNQED: Built-in Message Display Demo")
    print("=" * 80)
    print()
    print("This demo shows the NEW built-in colored message display feature.")
    print("No custom callbacks needed - just set display_messages=True!")
    print()
    print("─" * 80)
    print()
    
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        print("❌ Error: OPENAI_API_KEY not found in environment")
        print("   Please create a .env file with your OpenAI API key")
        return
    
    # Example 1: OrchestratedWorkspace with built-in display
    print("📋 Example 1: OrchestratedWorkspace")
    print("   Creating workspace with display_messages=True...\n")
    
    # For this demo, we'll use mock agents
    # In a real scenario, you would use actual agent instances
    
    print("✅ Setup complete!")
    print()
    print("Key features:")
    print("  • Automatic color assignment for each agent")
    print("  • Clean, formatted message display")
    print("  • No custom callback code needed")
    print("  • Works with both WorkspaceEngine and OrchestratedWorkspace")
    print()
    print("─" * 80)
    print()
    
    # Example 2: WorkspaceEngine with built-in display
    print("📋 Example 2: WorkspaceEngine (create_workspace)")
    print()
    print("Simple usage:")
    print()
    print("```python")
    print("workspace = synqed.create_workspace(")
    print("    name='My Workspace',")
    print("    task='Collaborate on a task',")
    print("    agents=[agent1, agent2, agent3],")
    print("    display_messages=True,  # Enable colored display")
    print(")")
    print("result = await workspace.run()")
    print("```")
    print()
    print("─" * 80)
    print()
    
    # Example 3: Verbose mode
    print("📋 Example 3: Verbose Mode")
    print()
    print("For more detailed output, use display_verbose=True:")
    print()
    print("```python")
    print("workspace = synqed.create_workspace(")
    print("    name='My Workspace',")
    print("    task='Collaborate on a task',")
    print("    agents=[agent1, agent2, agent3],")
    print("    display_messages=True,")
    print("    display_verbose=True,  # Show message type, phase, etc.")
    print(")")
    print("```")
    print()
    print("─" * 80)
    print()
    
    # Example 4: Combining with custom callbacks
    print("📋 Example 4: Custom Callbacks Still Work")
    print()
    print("You can still add your own callbacks for additional processing:")
    print()
    print("```python")
    print("def my_callback(msg):")
    print("    # Log to file, send to monitoring, etc.")
    print("    logger.info(f'{msg.sender_name}: {msg.content}')")
    print()
    print("workspace = synqed.create_workspace(")
    print("    name='My Workspace',")
    print("    task='Collaborate on a task',")
    print("    agents=[agent1, agent2, agent3],")
    print("    display_messages=True,  # Built-in display")
    print("    on_message=my_callback,  # Your custom logic")
    print(")")
    print("```")
    print()
    print("Both the built-in display AND your custom callback will be called!")
    print()
    print("=" * 80)
    print("  Demo Complete")
    print("=" * 80)
    print()
    print("✨ The built-in display feature makes it easy to see what's happening")
    print("   in your multi-agent workspace without writing any display code!")
    print()


if __name__ == "__main__":
    asyncio.run(main())

