"""
Synqed - A simplified wrapper around A2A for multi-agent systems.

This module provides high-level abstractions for creating, managing,
and coordinating AI agents using the A2A protocol.
"""

import asyncio
import aiohttp

from synqed.agent import Agent
from synqed.agent_card import AgentCardBuilder
from synqed.client import Client
from synqed.delegator import TaskDelegator
from synqed.orchestrator import (
    Orchestrator,
    LLMProvider,
    AgentSelection,
    OrchestrationResult,
)
from synqed.server import AgentServer
from synqed.workspace import (
    Workspace,
    WorkspaceState,
    WorkspaceMessage,
    WorkspaceArtifact,
    WorkspaceParticipant,
    MessageType,
)
from synqed.orchestrated_workspace import (
    OrchestratedWorkspace,
    ExecutionPlan,
    ExecutionResult,
    Subtask,
    CollaborationMode,
)

# New workspace engine - the intelligence layer
from synqed.workspace_engine import (
    WorkspaceEngine,
    WorkspaceConfig,
    WorkspaceLifecycleState,
    AgentInfo,
    WorkspaceStatistics,
)
from synqed.message_model import (
    StructuredMessage,
    MessageType as StructuredMessageType,
    MessagePriority,
    MessageMetadata,
    create_suggestion,
    create_critique,
    create_vote,
    create_final_output_proposal,
)
from synqed.protocols import (
    Protocol,
    ProtocolType,
    ProtocolPhase,
    ProtocolConfig,
    CritiqueAndReviseProtocol,
    DebateProtocol,
    ConsensusProtocol,
    create_protocol,
)
from synqed.scheduler import (
    Scheduler,
    SchedulerConfig,
    SchedulingStrategy,
    AgentTurnState,
)
from synqed.shared_state import (
    SharedState,
    StateEntry,
    AgentMemory,
)
from synqed.constraints import (
    ConstraintEnforcer,
    AgentConstraints,
    WorkspaceRules,
    ConstraintResult,
    create_default_constraints,
    create_workspace_rules,
)
from synqed.observability import (
    ObservabilityCollector,
    AgentContribution,
    InteractionGraph,
    ExecutionTrace,
    create_observability_report,
)
from synqed.builders import (
    create_workspace,  # THE primary API - use this!
    create_consensus_workspace,  # Legacy, prefer create_workspace
    create_debate_workspace,  # Legacy, prefer create_workspace
    create_critique_workspace,  # Legacy, prefer create_workspace
)
from synqed.display import (
    ColoredAgentDisplay,
    create_colored_printer,
    Colors,
)

__version__ = "1.0.7"

__all__ = [
    # === PRIMARY API ===
    # Use these for new code!
    "create_workspace",  # ⭐ THE main function - creates any workspace type
    "AgentInfo",  # Define agents
    "WorkspaceEngine",  # The workspace instance
    
    # === Display Utilities ===
    "ColoredAgentDisplay",  # Colored message display
    "create_colored_printer",  # Quick colored printer
    "Colors",  # ANSI color codes
    
    # === Core Components ===
    # Protocol types
    "ProtocolType",  # CONSENSUS, DEBATE, CRITIQUE_AND_REVISE
    "ProtocolPhase",
    "ProtocolConfig",
    # Scheduling
    "SchedulingStrategy",  # PROTOCOL_DRIVEN, ROUND_ROBIN, etc.
    "SchedulerConfig",
    # State management
    "SharedState",
    "StateEntry",
    "AgentMemory",
    # Lifecycle
    "WorkspaceLifecycleState",
    "WorkspaceStatistics",
    
    # === Message Model ===
    "StructuredMessage",
    "StructuredMessageType",
    "MessagePriority",
    "MessageMetadata",
    "create_suggestion",
    "create_critique",
    "create_vote",
    "create_final_output_proposal",
    
    # === Advanced Features ===
    # Protocols (for advanced use)
    "Protocol",
    "CritiqueAndReviseProtocol",
    "DebateProtocol",
    "ConsensusProtocol",
    "create_protocol",
    # Scheduler (for advanced use)
    "Scheduler",
    "AgentTurnState",
    # Constraints
    "ConstraintEnforcer",
    "AgentConstraints",
    "WorkspaceRules",
    "ConstraintResult",
    "create_default_constraints",
    "create_workspace_rules",
    # Observability
    "ObservabilityCollector",
    "AgentContribution",
    "InteractionGraph",
    "ExecutionTrace",
    "create_observability_report",
    
    # === Legacy/Backwards Compatibility ===
    # Workspace config (prefer using create_workspace parameters)
    "WorkspaceConfig",
    # Legacy workspace functions (prefer create_workspace)
    "create_consensus_workspace",
    "create_debate_workspace",
    "create_critique_workspace",
    # Legacy A2A wrappers (for compatibility)
    "Agent",
    "AgentCardBuilder",
    "AgentServer",
    "Client",
    "TaskDelegator",
    "Orchestrator",
    "LLMProvider",
    "AgentSelection",
    "OrchestrationResult",
    "Workspace",
    "WorkspaceState",
    "WorkspaceMessage",
    "WorkspaceArtifact",
    "WorkspaceParticipant",
    "MessageType",
    "OrchestratedWorkspace",
    "ExecutionPlan",
    "ExecutionResult",
    "Subtask",
    "CollaborationMode",
    
    # Utils
    "asyncio",
    "aiohttp",
]

