"""
End-to-end tests for Synqed.

These tests simulate real-world usage scenarios from start to finish.
"""

import pytest
import asyncio
from synqed import Agent, AgentServer, Client, TaskDelegator


class TestEndToEnd:
    """End-to-end test scenarios."""
    
    @pytest.mark.asyncio
    async def test_simple_conversation_flow(self, port_generator):
        """
        E2E Test: Simple conversation between user and agent.
        
        Scenario:
        1. Developer creates an agent
        2. Starts a server
        3. User connects via client
        4. User has a conversation with the agent
        """
        # Developer creates an agent
        conversation_history = []
        
        async def conversational_executor(context):
            message = context.get_user_input()
            conversation_history.append(message)
            
            if "hello" in message.lower():
                return "Hello! How can I help you today?"
            elif "help" in message.lower():
                return "I can assist you with various tasks. What do you need?"
            else:
                return f"You said: {message}. Anything else I can help with?"
        
        agent = Agent(
            name="Helpful Assistant",
            description="A conversational assistant",
            skills=["conversation", "help"],
            executor=conversational_executor
        )
        
        # Start server
        port = port_generator()
        server = AgentServer(agent, host="127.0.0.1", port=port)
        await server.start_background()
        await asyncio.sleep(1)
        
        try:
            # User connects
            async with Client(agent_url=server.url) as client:
                # Conversation
                response1 = await client.ask("Hello there")
                assert "Hello" in response1
                assert "help" in response1.lower()
                
                response2 = await client.ask("I need help")
                assert "assist" in response2.lower() or "help" in response2.lower()
                
                response3 = await client.ask("Thank you")
                assert "Thank you" in response3
            
            # Verify conversation was tracked
            assert len(conversation_history) == 3
        finally:
            await server.stop()
    
    @pytest.mark.asyncio
    async def test_multi_agent_collaboration_system(self, port_generator):
        """
        E2E Test: Multi-agent system collaborating on complex task.
        
        Scenario:
        1. Create multiple specialized agents (Research, Writing, Review)
        2. Set up task delegator
        3. Submit complex task that requires multiple agents
        4. Verify proper delegation and results
        """
        # Create specialized agents
        async def research_executor(context):
            message = context.get_user_input()
            return f"Research findings on: {message}\n- Fact 1\n- Fact 2\n- Fact 3"
        
        async def writing_executor(context):
            message = context.get_user_input()
            return f"Article based on: {message}\n\nIntroduction...\nBody...\nConclusion..."
        
        async def review_executor(context):
            message = context.get_user_input()
            return f"Review of: {message}\n\nQuality: Good\nSuggestions: Minor edits needed"
        
        research_agent = Agent(
            name="Research Agent",
            description="Conducts research",
            skills=["research", "analysis"],
            executor=research_executor
        )
        
        writing_agent = Agent(
            name="Writing Agent",
            description="Writes content",
            skills=["writing", "content"],
            executor=writing_executor
        )
        
        review_agent = Agent(
            name="Review Agent",
            description="Reviews content",
            skills=["review", "editing"],
            executor=review_executor
        )
        
        # Start servers
        ports = [port_generator() for _ in range(3)]
        servers = [
            AgentServer(research_agent, host="127.0.0.1", port=ports[0]),
            AgentServer(writing_agent, host="127.0.0.1", port=ports[1]),
            AgentServer(review_agent, host="127.0.0.1", port=ports[2]),
        ]
        
        for server in servers:
            await server.start_background()
        await asyncio.sleep(1)
        
        try:
            # Set up delegator
            delegator = TaskDelegator()
            delegator.register_agent(agent=research_agent)
            delegator.register_agent(agent=writing_agent)
            delegator.register_agent(agent=review_agent)
            
            # Simulate workflow: Research -> Write -> Review
            topic = "Climate Change"
            
            # Step 1: Research
            research_result = await delegator.submit_task(
                f"Research {topic}",
                require_skills=["research"]
            )
            assert "Research findings" in research_result
            assert topic in research_result
            
            # Step 2: Write
            writing_result = await delegator.submit_task(
                f"Write article about {topic}",
                require_skills=["writing"]
            )
            assert "Article" in writing_result
            
            # Step 3: Review
            review_result = await delegator.submit_task(
                f"Review article on {topic}",
                require_skills=["review"]
            )
            assert "Review" in review_result
            
            await delegator.close_all()
        finally:
            for server in servers:
                await server.stop()
    
    @pytest.mark.asyncio
    async def test_agent_discovery_and_dynamic_routing(self, port_generator):
        """
        E2E Test: Dynamic agent discovery and routing.
        
        Scenario:
        1. Multiple agents are running
        2. Delegator discovers them
        3. User submits various tasks
        4. System automatically routes to appropriate agent
        """
        # Create diverse agents
        agents_config = [
            ("Calculator", ["math", "calculation"], lambda ctx: "Result: 42"),
            ("Translator", ["translation", "language"], lambda ctx: "Translated: Bonjour"),
            ("Timer", ["time", "scheduling"], lambda ctx: "Timer set for 5 minutes"),
        ]
        
        agents = []
        servers = []
        ports = []
        
        for name, skills, func in agents_config:
            async def executor(context, f=func):
                return f(context)
            
            agent = Agent(
                name=name,
                description=f"{name} agent",
                skills=skills,
                executor=executor
            )
            agents.append(agent)
            
            port = port_generator()
            ports.append(port)
            server = AgentServer(agent, host="127.0.0.1", port=port)
            servers.append(server)
            await server.start_background()
        
        await asyncio.sleep(1)
        
        try:
            # Set up delegator with all agents
            delegator = TaskDelegator()
            for agent in agents:
                delegator.register_agent(agent=agent)
            
            # List available agents
            available_agents = delegator.list_agents()
            assert len(available_agents) == 3
            
            # Submit tasks and verify routing
            math_result = await delegator.submit_task(
                "Calculate 20 + 22",
                require_skills=["math"]
            )
            assert "42" in math_result
            
            translation_result = await delegator.submit_task(
                "Translate Hello to French",
                require_skills=["translation"]
            )
            assert "Bonjour" in translation_result
            
            timer_result = await delegator.submit_task(
                "Set a timer",
                require_skills=["time"]
            )
            assert "Timer" in timer_result
            
            await delegator.close_all()
        finally:
            for server in servers:
                await server.stop()
    
    @pytest.mark.asyncio
    async def test_streaming_conversation(self, port_generator):
        """
        E2E Test: Streaming conversation with real-time responses.
        
        Scenario:
        1. Create agent that can stream responses
        2. Client connects with streaming enabled
        3. Verify responses stream in real-time
        """
        async def streaming_executor(context):
            message = context.get_user_input()
            # Simulate generating response in parts
            return f"Processing your request: {message}. Here's the result!"
        
        agent = Agent(
            name="Streaming Agent",
            description="Supports streaming",
            skills=["streaming"],
            executor=streaming_executor,
            capabilities={"streaming": True}
        )
        
        port = port_generator()
        server = AgentServer(agent, host="127.0.0.1", port=port)
        await server.start_background()
        await asyncio.sleep(1)
        
        try:
            async with Client(agent_url=server.url, streaming=True) as client:
                # Collect streamed chunks
                chunks = []
                async for chunk in client.stream("Tell me a story"):
                    chunks.append(chunk)
                
                # Should receive response
                full_response = "".join(chunks)
                assert len(full_response) > 0
                assert "Processing" in full_response or "story" in full_response
        finally:
            await server.stop()
    
    @pytest.mark.asyncio
    async def test_error_handling_and_recovery(self, port_generator):
        """
        E2E Test: Error handling and graceful recovery.
        
        Scenario:
        1. Create agent that can fail
        2. Test various error scenarios
        3. Verify system handles errors gracefully
        """
        class ProcessingError(Exception):
            pass
        
        async def unreliable_executor(context):
            message = context.get_user_input()
            
            if "fail" in message.lower():
                raise ProcessingError("Simulated failure")
            
            return f"Success: {message}"
        
        agent = Agent(
            name="Unreliable Agent",
            description="Sometimes fails",
            skills=["test"],
            executor=unreliable_executor
        )
        
        port = port_generator()
        server = AgentServer(agent, host="127.0.0.1", port=port)
        await server.start_background()
        await asyncio.sleep(1)
        
        try:
            async with Client(agent_url=server.url) as client:
                # Successful request
                success_response = await client.ask("This should work")
                assert "Success" in success_response
                
                # Note: Error handling depends on A2A protocol error handling
                # The agent may return an error message rather than raising
        finally:
            await server.stop()
    
    @pytest.mark.asyncio
    async def test_complete_workflow_from_scratch(self, port_generator):
        """
        E2E Test: Complete workflow from scratch.
        
        This test simulates a complete user journey:
        1. Developer builds an agent system from zero
        2. Deploys multiple agents
        3. User interacts with the system
        4. System handles complex multi-step tasks
        """
        # Step 1: Developer defines agent logic
        class CustomerSupportSystem:
            def __init__(self):
                self.ticket_db = {}
                self.ticket_counter = 0
            
            async def create_ticket(self, context):
                message = context.get_user_input()
                self.ticket_counter += 1
                ticket_id = f"TICKET-{self.ticket_counter}"
                self.ticket_db[ticket_id] = message
                return f"Ticket created: {ticket_id}"
            
            async def check_status(self, context):
                message = context.get_user_input()
                # Extract ticket ID (simplified)
                for ticket_id in self.ticket_db:
                    if ticket_id in message:
                        return f"Ticket {ticket_id} status: Open"
                return "Ticket not found"
        
        support_system = CustomerSupportSystem()
        
        # Step 2: Create agents
        ticket_agent = Agent(
            name="Ticket Agent",
            description="Creates support tickets",
            skills=["tickets", "support"],
            executor=support_system.create_ticket
        )
        
        status_agent = Agent(
            name="Status Agent",
            description="Checks ticket status",
            skills=["status", "tracking"],
            executor=support_system.check_status
        )
        
        # Step 3: Deploy agents
        ports = [port_generator() for _ in range(2)]
        
        ticket_server = AgentServer(ticket_agent, host="127.0.0.1", port=ports[0])
        status_server = AgentServer(status_agent, host="127.0.0.1", port=ports[1])
        
        await ticket_server.start_background()
        await status_server.start_background()
        await asyncio.sleep(1)
        
        try:
            # Step 4: User interacts with system
            delegator = TaskDelegator()
            delegator.register_agent(agent=ticket_agent)
            delegator.register_agent(agent=status_agent)
            
            # Create a ticket
            create_result = await delegator.submit_task(
                "I need help with my account",
                require_skills=["tickets"]
            )
            assert "TICKET" in create_result
            
            # Check ticket status
            status_result = await delegator.submit_task(
                "Check status of TICKET-1",
                require_skills=["status"]
            )
            assert "status" in status_result.lower()
            
            # Verify system state
            assert len(support_system.ticket_db) == 1
            
            await delegator.close_all()
        finally:
            await ticket_server.stop()
            await status_server.stop()

