"""For functions that directly or indirectly help to for rendering ELN.
Note that this not schema eln that is rendered to Nomad rather the eln that
is generated by schema eln."""
# Copyright The NOMAD Authors.
#
# This file is part of NOMAD. See https://nomad-lab.eu for further info.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import re

from pynxtools.dataconverter.nexus_tree import NexusEntity, NexusGroup, NexusNode
from pynxtools.eln_mapper.eln import ElnGenerator


class ReaderElnGenerator(ElnGenerator):
    """Class for creating YAML files to be used directly in the pynxtools dataconverter."""

    def _generate_output_file_name(self, output_file: str):
        """
        Generate the output file name of the schema ELN generator.

        The output file name will be:
        - <output_file>.eln_data.yaml or
        - <output_file> if output_file already ends on eln_data.yaml

        If no output_file is given, the output will be <nxdl_name>.eln_data.yaml,
        where <nxdl_name> is the name of the application definition without the leading NX
        (e.g., for NXmpes, the file is called mpes.scheme.archive.yaml).

        """
        file_parts: list = []
        out_file_ext = "eln_data.yaml"
        raw_name = ""
        out_file = ""

        if self.output_file is None:
            out_file = ".".join([self.nxdl[2:], out_file_ext])
        else:
            if output_file.endswith(out_file_ext):
                out_file = self.output_file
            else:
                file_parts = output_file.split(".")
                if len(file_parts) == 1:
                    raw_name = file_parts[0]
                    out_file = ".".join([raw_name, out_file_ext])
                elif len(file_parts) == 2 and ".".join(file_parts[1:]) == out_file_ext:
                    out_file = output_file
                else:
                    raise ValueError(
                        "Check for correct NeXus definition and output file name."
                    )

        return out_file

    def _construct_group_structure(
        self, node: NexusGroup, recursive_dict: dict, recursion_level: int
    ) -> None:
        """Handle NeXus group, to construct group structure as follows:
        <group_name>:
            section:
                m_annotations:
                    eln:
                        overview: true

        Parameters
        ----------
        node: NexusGroup
            NeXus group to recurse
        recursive_dict : dict
            dict into which the group is recursively added
        recursion_level: int
            Recursion level in the tree, used to (optionally) skip upper levels like NXentry
        """
        if not super()._construct_group_structure(
            node, recursive_dict, recursion_level
        ):
            return

        group_name = node.name

        if node.variadic:
            # TODO: allow variadic names?!
            group_name = group_name.lower()

        if not node.children:
            recursive_dict[group_name] = None
            return

        recursive_dict[group_name] = {}

        # pass the grp elment for recursive search
        self._recurse_tree(node, recursive_dict[group_name], recursion_level + 1)

    def _construct_entity_structure(
        self, node: NexusEntity, recursive_dict: dict, recursion_level: int
    ):
        """Handle NeXus field or attribute, to construct structure like:
        <entity_name>:
            type: np.float64 (matching with the node's type)
            unit: <some-unit> (matching with the node's unit)
            m_annotations:
                eln:
                component: NumberEditQuantity (matching with the node's type)
                defaultDisplayUnit: <some-unit> (matching with the node's unit)
            description: node docs

        Parameters
        ----------
        node: NexusEntity
            NeXus field/attribute to recurse
        recursive_dict : dict
            dict into which the entity is recursively added
        recursion_level: int
            Recursion level in the tree, used to (optionally) skip upper levels like NXentry
        """

        if not super()._construct_entity_structure(
            node, recursive_dict, recursion_level
        ):
            return

        entity_name = node.name.lower()

        if node.nx_type == "attribute":
            entity_name = f"/@{entity_name}"

        if not node.children and not node.unit:
            recursive_dict[entity_name] = None
            return

        recursive_dict[entity_name] = {}

        if node.unit:
            recursive_dict[entity_name].update({"value": None, "unit": None})

        self._recurse_tree(node, recursive_dict[entity_name], recursion_level + 1)
