"""Core scaffolding logic for Claude Code configurations."""

import json
from pathlib import Path
from typing import Any

import yaml

from claude_setup.agents.generator import ConfigGenerator
from claude_setup.config import ClaudeConfig, ProjectConfig, merge_configs
from claude_setup.templates.manager import TemplateManager
from claude_setup.utils.detector import ProjectDetector


class Scaffolder:
    """Main scaffolding engine for Claude Code configurations."""

    def __init__(self, project_path: Path):
        """Initialize scaffolder with project path.

        Args:
            project_path: Path to the project root
        """
        self.project_path = Path(project_path)
        self.claude_dir = self.project_path / ".claude"
        self.template_manager = TemplateManager()
        self.config_generator = ConfigGenerator()

    def create_structure(self) -> None:
        """Create the basic .claude directory structure."""
        # Create directories
        self.claude_dir.mkdir(exist_ok=True)
        (self.claude_dir / "commands").mkdir(exist_ok=True)
        (self.claude_dir / "agents").mkdir(exist_ok=True)
        (self.claude_dir / "hooks").mkdir(exist_ok=True)
        (self.claude_dir / "mcp").mkdir(exist_ok=True)

        # Create .gitignore
        gitignore_content = """# Claude Setup generated files
settings.local.json
*.log
*.tmp
.cache/
__pycache__/
*.pyc
.DS_Store
"""
        (self.claude_dir / ".gitignore").write_text(gitignore_content)

    def generate_base_config(self) -> None:
        """Generate base configuration files."""
        # Base settings.json
        settings = {
            "model": "claude-opus-4-1-20250805",
            "permissions": {
                "allow": [
                    "Read(**/*)",
                    "Write(**/*)",
                    "Bash(git:*)",
                    "Bash(ls:*)",
                    "Bash(cat:*)",
                    "Bash(grep:*)",
                    "Bash(find:*)",
                ],
                "deny": [
                    "Read(.env*)",
                    "Read(**/*.key)",
                    "Read(**/*.pem)",
                    "Delete(**)",
                    "Bash(rm:*)",
                    "Bash(sudo:*)",
                ],
            },
            "env": {"PROJECT_INITIALIZED": "true", "CLAUDE_SETUP_VERSION": "1.0.0"},
        }

        self._save_json(self.claude_dir / "settings.json", settings)

        # Base CLAUDE.md if it doesn't exist
        claude_md_path = self.project_path / "CLAUDE.md"
        if not claude_md_path.exists():
            claude_md = """# Project Configuration

This project uses Claude Code for AI-assisted development.

## Quick Start

1. Run `/help` to see available commands
2. Use `/configure` to customize settings
3. Start coding with Claude's assistance

## Project Structure

Describe your project structure here.

## Development Guidelines

Add your coding standards and practices.

## Available Commands

Custom commands are in `.claude/commands/`

## Agents

Specialized agents are in `.claude/agents/`

---
*Generated by claude-setup v1.0.0*
"""
            claude_md_path.write_text(claude_md)

    def apply_template(self, template_name: str) -> None:
        """Apply a template to the project.

        Args:
            template_name: Name of the template to apply
        """
        template_data = self.template_manager.get_template(template_name)
        self.apply_template_data(template_data)

    def apply_template_data(self, template_data: dict[str, Any]) -> None:
        """Apply template data to the project.

        Args:
            template_data: Template configuration dictionary
        """
        # Apply settings
        if "settings" in template_data:
            settings_path = self.claude_dir / "settings.json"
            existing = self._load_json(settings_path) if settings_path.exists() else {}
            merged = merge_configs(existing, template_data["settings"])
            self._save_json(settings_path, merged)

        # Apply CLAUDE.md
        if "claude_md" in template_data:
            (self.project_path / "CLAUDE.md").write_text(template_data["claude_md"])

        # Apply commands
        if "commands" in template_data:
            for name, content in template_data["commands"].items():
                (self.claude_dir / "commands" / f"{name}.md").write_text(content)

        # Apply agents
        if "agents" in template_data:
            for name, content in template_data["agents"].items():
                (self.claude_dir / "agents" / f"{name}.md").write_text(content)

        # Apply hooks
        if "hooks" in template_data:
            self._apply_hooks(template_data["hooks"])

        # Apply MCP servers
        if "mcp_servers" in template_data:
            self._apply_mcp_servers(template_data["mcp_servers"])

    def apply_configuration(self, config: ProjectConfig) -> None:
        """Apply a project configuration.

        Args:
            config: Project configuration object
        """
        # Generate optimized configuration
        claude_config = self.config_generator.generate(config)

        # Apply the generated configuration
        self.apply_claude_config(claude_config)

        # Save project config for future reference
        from claude_setup.config import save_config

        save_config(self.project_path, config)

    def apply_claude_config(self, config: ClaudeConfig) -> None:
        """Apply a complete Claude configuration.

        Args:
            config: Claude configuration object
        """
        # Apply settings
        if config.settings:
            settings_path = self.claude_dir / "settings.json"
            existing = self._load_json(settings_path) if settings_path.exists() else {}
            merged = merge_configs(existing, config.settings)
            self._save_json(settings_path, merged)

        # Apply CLAUDE.md
        if config.claude_md:
            (self.project_path / "CLAUDE.md").write_text(config.claude_md)

        # Apply commands
        if config.commands:
            for name, content in config.commands.items():
                command_path = self.claude_dir / "commands" / f"{name}.md"
                command_path.parent.mkdir(exist_ok=True)
                command_path.write_text(content)

        # Apply agents
        if config.agents:
            for name, content in config.agents.items():
                agent_path = self.claude_dir / "agents" / f"{name}.md"
                agent_path.parent.mkdir(exist_ok=True)
                agent_path.write_text(content)

        # Apply hooks
        if config.hooks:
            self._apply_hooks(config.hooks)

        # Apply MCP servers
        if config.mcp_servers:
            self._apply_mcp_servers(config.mcp_servers)

    def auto_detect(self) -> ProjectConfig:
        """Auto-detect project configuration.

        Returns:
            Detected project configuration
        """
        detector = ProjectDetector(self.project_path)
        detected = detector.detect()
        return ProjectConfig(**detected)

    def validate_configuration(self) -> list[str]:
        """Validate current configuration.

        Returns:
            List of validation issues (empty if valid)
        """
        issues = []

        # Check required files
        if not (self.claude_dir / "settings.json").exists():
            issues.append("Missing settings.json")

        if not (self.project_path / "CLAUDE.md").exists():
            issues.append("Missing CLAUDE.md")

        # Validate settings.json
        try:
            settings = self._load_json(self.claude_dir / "settings.json")
            if "permissions" not in settings:
                issues.append("settings.json missing 'permissions' field")
            if "model" not in settings:
                issues.append("settings.json missing 'model' field")
        except Exception as e:
            issues.append(f"Invalid settings.json: {e}")

        # Check for at least one command or agent
        commands = list((self.claude_dir / "commands").glob("*.md"))
        agents = list((self.claude_dir / "agents").glob("*.md"))

        if not commands and not agents:
            issues.append("No commands or agents defined")

        return issues

    # Private helper methods

    def _apply_hooks(self, hooks: dict[str, Any]) -> None:
        """Apply hook configurations."""
        hooks_dir = self.claude_dir / "hooks"
        hooks_dir.mkdir(exist_ok=True)

        for event, hook_configs in hooks.items():
            hook_file = hooks_dir / f"{event.lower()}.json"
            self._save_json(hook_file, hook_configs)

    def _apply_mcp_servers(self, servers: list[dict[str, Any]]) -> None:
        """Apply MCP server configurations."""
        mcp_dir = self.claude_dir / "mcp"
        mcp_dir.mkdir(exist_ok=True)

        mcp_config = {"mcpServers": servers}

        self._save_json(mcp_dir / "config.json", mcp_config)

    def _save_json(self, path: Path, data: dict) -> None:
        """Save JSON data to file."""
        path.parent.mkdir(parents=True, exist_ok=True)
        with open(path, "w") as f:
            json.dump(data, f, indent=2)

    def _load_json(self, path: Path) -> dict:
        """Load JSON data from file."""
        with open(path) as f:
            return json.load(f)

    def _save_yaml(self, path: Path, data: dict) -> None:
        """Save YAML data to file."""
        path.parent.mkdir(parents=True, exist_ok=True)
        with open(path, "w") as f:
            yaml.dump(data, f, default_flow_style=False, sort_keys=False)

    def _load_yaml(self, path: Path) -> dict:
        """Load YAML data from file."""
        with open(path) as f:
            return yaml.safe_load(f)
