---
sidebar_position: 3
---

# Core Concepts

Understanding mem8's key concepts and architecture.

## Overview

mem8 is built around three core concepts:
1. **Thoughts** - Structured markdown documentation
2. **Templates** - Project initialization and standardization
3. **Claude Code Integration** - AI-assisted development workflows

## Thoughts

### What are Thoughts?

Thoughts are structured markdown documents that capture:
- **Research** - Codebase analysis and investigation
- **Plans** - Implementation designs and steps
- **Decisions** - Technical choices and rationale
- **PRs** - Pull request descriptions and context
- **Notes** - General observations and ideas

### Thought Structure

```
thoughts/
├── shared/          # Team-shared thoughts
│   ├── research/    # Research documents
│   ├── plans/       # Implementation plans
│   ├── prs/         # PR descriptions
│   └── decisions/   # Technical decisions
├── {username}/      # Personal thoughts
│   ├── notes/       # Personal notes
│   └── tickets/     # Work items
└── searchable/      # Symlinks for fast search
```

### Shared vs Personal

- **Shared** (`thoughts/shared/`) - Committed to git, visible to team
- **Personal** (`thoughts/{username}/`) - Your private workspace
- **Searchable** - Automatic symlinks for unified search

### Thought Lifecycle

1. **Create** - Use slash commands (`/research_codebase`, `/create_plan`)
2. **Edit** - Modify markdown files directly
3. **Search** - Find with `mem8 search "query"`
4. **Sync** - Share with team via `mem8 sync`

## Templates

### Template System

mem8 uses [Cookiecutter](https://cookiecutter.readthedocs.io/) for project templating with three main templates:

#### claude-config
Generates Claude Code customizations:
- **Agents** - Specialized AI helpers
- **Commands** - Workflow automation
- **Configuration** - Workspace settings

#### thoughts-repo
Creates thought repository structure:
- Directory hierarchy
- README and documentation
- Sync scripts

#### full
Combines both claude-config and thoughts-repo.

### External Templates

mem8 supports templates from:
- **GitHub** - `org/repo` or full URLs
- **Git** - Any git repository
- **Local** - Filesystem paths
- **Builtin** - Packaged with mem8

See [External Templates](./external-templates) for details.

### Template Manifest

Templates can include a `mem8-templates.yaml` manifest:

```yaml
version: 1
source: "."

templates:
  my-template:
    path: "template-dir"
    type: "cookiecutter"
    description: "Template description"
    variables:
      key: "default value"
```

## Claude Code Integration

### What is Claude Code?

[Claude Code](https://claude.ai/code) is Anthropic's AI-powered development environment. mem8 enhances it with:

### Agents

Specialized AI assistants for:
- **codebase-analyzer** - Deep code analysis
- **thoughts-locator** - Find relevant thoughts
- **web-search-researcher** - Research online
- **github-workflow-agent** - GitHub automation

### Commands

Workflow automation:
- **`/research_codebase`** - Analyze existing code
- **`/create_plan`** - Design implementation
- **`/implement_plan`** - Execute with context
- **`/commit`** - Create conventional commits
- **`/describe_pr`** - Generate PR descriptions

### Workflow Integration

mem8 connects Claude Code to:
- **GitHub** - Issues, PRs, workflows
- **Thoughts** - Research and plans
- **Templates** - Standardized setups

## Workspace

### Workspace Structure

A mem8 workspace contains:

```
project/
├── .mem8/            # Workspace metadata
│   └── toolbelt.json # CLI toolbelt manifest
├── .claude/          # Claude Code customizations
│   ├── agents/      # AI agent definitions
│   ├── commands/    # Slash command scripts
│   └── CLAUDE.md    # Workspace configuration
├── thoughts/         # Thought repository
│   ├── shared/      # Team thoughts
│   └── username/    # Personal thoughts
└── README.md         # Project documentation
```

The `.mem8` directory stores lightweight workspace metadata. By default it contains `toolbelt.json`, a manifest generated by your template that lists core, recommended, and optional CLI tools for Windows, macOS, and Linux. `mem8 doctor` reads this file to highlight missing tooling without installing anything automatically.

### Workspace Initialization

```bash
# Interactive setup
mem8 init

# Specify template
mem8 init --template claude-config

# Use external templates
mem8 init --template-source org/repo
```

### Workspace Health

Check workspace status:

```bash
# Quick check
mem8 status

# Detailed diagnostics
mem8 status --detailed

# Auto-fix issues
mem8 doctor --auto-fix
```

## Configuration

### Config Locations

mem8 uses platformdirs for config storage:

- **Config** - `~/.config/mem8/config.yaml`
- **Data** - `~/.local/share/mem8/`
- **Cache** - `~/.cache/mem8/`

### Config File

Example `~/.config/mem8/config.yaml`:

```yaml
workspace:
  default_template: full
  auto_sync: true

templates:
  default_source: org/custom-templates

discovery:
  cross_repo: false

workflow:
  provider: github
  automation_level: standard
```

### Environment Variables

Override config with environment variables:

```bash
# Override config directory
export MEM8_CONFIG_DIR=/custom/path

# Override data directory
export MEM8_DATA_DIR=/custom/data
```

## Search and Discovery

### Search System

mem8 provides fast full-text search:

```bash
# Basic search
mem8 search "authentication"

# Limit results
mem8 search "query" --limit 10

# Specific locations
mem8 search "query" --path thoughts/shared/plans
```

### Search Indexing

- **Automatic** - Indexes thoughts on creation
- **Incremental** - Updates on file changes
- **Fast** - Uses searchable symlinks

### Discovery Modes

- **Single-repo** - Search current workspace only
- **Cross-repo** - Search across all workspaces (opt-in)

## Sync and Collaboration

### Sync Strategies

```bash
# Two-way sync
mem8 sync --direction both

# Push only
mem8 sync --direction push

# Pull only
mem8 sync --direction pull

# Dry run
mem8 sync --dry-run
```

### Conflict Resolution

- **Prompt** - Ask user (default)
- **Local** - Prefer local changes
- **Shared** - Prefer shared changes
- **Newest** - Use newest timestamp

### Team Workflows

1. **Initialize** - Team lead runs `mem8 init --template full`
2. **Share** - Commit `.claude/` and `thoughts/` to git
3. **Clone** - Team members clone repo
4. **Sync** - Use `mem8 sync` for thought sharing
5. **Search** - Everyone can search shared thoughts

## Architecture

### CLI Layer

Built with [Typer](https://typer.tiangolo.com/):
- Command-line interface
- Interactive prompts
- Rich output formatting

### Core Layer

Python modules:
- **config.py** - Configuration management
- **template_source.py** - Template resolution
- **discovery.py** - Thought discovery
- **sync.py** - Synchronization logic

### Storage Layer

- **Local** - Markdown files on filesystem
- **Database** - Optional PostgreSQL for web interface
- **Git** - Version control for thoughts

### Optional Web Layer

- **Backend** - FastAPI server
- **Frontend** - Next.js dashboard
- **Database** - PostgreSQL for persistence

## Design Principles

### CLI-First

Everything accessible via command line. Web interface is optional enhancement.

### Plain Text

Thoughts are markdown files. No lock-in, works with any editor.

### Git-Native

Leverages git for versioning, sync, and collaboration.

### Extensible

Templates and workflows are customizable. External sources supported.

### Team-Friendly

Shared thoughts and standardized workflows for collaboration.

## Next Steps

- **[User Guide](./user-guide/getting-started)** - Learn to use mem8
- **[External Templates](./external-templates)** - Customize templates
- **[GitHub](https://github.com/killerapp/mem8)** - Explore source code
