"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Activity = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const util_1 = require("./util");
class Activity extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.isVPCCustomized = false;
        this.isKMSCustomized = false;
        // Validate passed values for multi value replacements
        if (props.athenaWorkgroupNames && props.athenaWorkgroupNames.length === 0) {
            throw TypeError(`The value of the athenaWorkgroupNames parameter should be of type ${props.athenaWorkgroupNames} with at least one element.`);
        }
        for (const athenaWorkGroupName of props.athenaWorkgroupNames || []) {
            if (!util_1.parameterValidationRegex.test(athenaWorkGroupName)) {
                throw TypeError(`The value ${athenaWorkGroupName} of athenaWorkgroupNames array element contains an invalid character,
             athenaWorkgroupNames elements must be alphanumeric with no spaces and only the special characters: _/:.-`);
            }
        }
        if (props.ecrRepositories && props.ecrRepositories.length === 0) {
            throw TypeError(`The value of the ecrRepositories parameter should be of type ${props.ecrRepositories} with at least one element.`);
        }
        if (props.glueDatabaseNames && props.glueDatabaseNames.length === 0) {
            throw TypeError(`The value of the glueDatabaseNames parameter should be of type ${props.glueDatabaseNames} with at least one element.`);
        }
        for (const glueDatabaseName of props.glueDatabaseNames || []) {
            if (!util_1.parameterValidationRegex.test(glueDatabaseName)) {
                throw TypeError(`The value ${glueDatabaseName} of glueDatabaseNames array element contains an invalid character,
             glueDatabaseNames elements must be alphanumeric with no spaces and only the special characters: _/:.-`);
            }
        }
        if (props.rolesToPass && props.rolesToPass.length === 0) {
            throw TypeError(`The value of the rolesToPass parameter should be of type ${props.rolesToPass} with at least one element.`);
        }
        if (props.s3Buckets && props.s3Buckets.length === 0) {
            throw TypeError(`The value of the s3Buckets parameter should be of type ${props.s3Buckets} with at least one element.`);
        }
        this.activityName = props.activityName;
        this.version = props.version || 1;
        // Set single value replacements
        this.singleValueReplacements = new Map();
        this.singleValueReplacements.set(Activity.ACCOUNT_ID_PARAMETER_NAME, cdk.Token.isUnresolved(cdk.Stack.of(this).account) ? '*' : cdk.Stack.of(this).account);
        this.singleValueReplacements.set(Activity.REGION_PARAMETER_NAME, cdk.Token.isUnresolved(cdk.Stack.of(this).region) ? '*' : cdk.Stack.of(this).region);
        // Set multi value replacements
        this.multiValueReplacements = new Map();
        this.multiValueReplacements.set(Activity.ATHENA_WORKGROUP_NAMES_PARAMETER_NAME, props.athenaWorkgroupNames ?? []);
        this.multiValueReplacements.set(Activity.ECR_REPOSITORIES_PARAMETER_NAME, (props.ecrRepositories ?? []).map((ecrRepository) => ecrRepository.repositoryArn));
        this.multiValueReplacements.set(Activity.GLUE_DATABASE_NAMES_PARAMETER_NAME, props.glueDatabaseNames ?? []);
        this.multiValueReplacements.set(Activity.PASSED_ROLES_PARAMETER_NAME, (props.rolesToPass ?? []).map((role) => role.roleArn));
        this.multiValueReplacements.set(Activity.S3_BUCKETS_PARAMETER_NAME, (props.s3Buckets ?? []).map((s3Bucket) => s3Bucket.bucketName));
        this.isCustomizationAvailableForVPC = props.isCustomizationAvailableForVPC;
        this.isCustomizationAvailableForKMS = props.isCustomizationAvailableForKMS;
    }
    // Activity Selection Functions
    static accessAwsServices(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.ACCESS_AWS_SERVICES,
            isCustomizationAvailableForVPC: true,
            isCustomizationAvailableForKMS: true,
            ecrRepositories: options.ecrRepositories,
            s3Buckets: options.s3Buckets,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static accessS3AllResources(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.ACCESS_S3_ALL_RESOURCES,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static accessS3AllResourcesV2(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.ACCESS_S3_ALL_RESOURCES,
            version: 2,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static accessS3Buckets(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.ACCESS_S3_BUCKETS,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
            s3Buckets: options.s3Buckets,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static manageEndpoints(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_ENDPOINTS_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: true,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static manageExperiments(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_EXPERIMENTS_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static manageGlueTables(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_GLUE_TABLES_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: true,
            s3Buckets: options.s3Buckets,
            glueDatabaseNames: options.glueDatabaseNames,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static manageJobs(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_JOBS_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: true,
            isCustomizationAvailableForKMS: true,
            rolesToPass: options.rolesToPass,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static manageModels(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_MODELS_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: true,
            isCustomizationAvailableForKMS: false,
            rolesToPass: options.rolesToPass,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static managePipelines(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MANAGE_PIPELINES_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
            rolesToPass: options.rolesToPass,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static monitorModels(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.MONITOR_MODELS_ACTIVITY_NAME,
            isCustomizationAvailableForVPC: true,
            isCustomizationAvailableForKMS: true,
            rolesToPass: options.rolesToPass,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static queryAthenaGroups(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.QUERY_ATHENA_WORKGROUPS,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
            athenaWorkgroupNames: options.athenaWorkgroupNames,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static runStudioApps(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.RUN_STUDIO_APPS,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: true,
            rolesToPass: options.rolesToPass,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static runStudioAppsV2(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.RUN_STUDIO_APPS,
            version: 2,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    static visualizeExperiments(scope, id, options) {
        const activity = new Activity(scope, id, {
            activityName: Activity.VISUALIZE_EXPERIMENTS,
            isCustomizationAvailableForVPC: false,
            isCustomizationAvailableForKMS: false,
        });
        activity.customizeVPC(options.subnets, options.securityGroups);
        activity.customizeKMS(options.dataKeys, options.volumeKeys);
        return activity;
    }
    /**
     * Get grant options from statement for Grant.addToPrincipal() function
     * @param identity identity to be granted permissions
     * @param statement the statement from which to get the actions, resources and conditions
     * @returns - The options for the Grant.addToPrincipal() function
     */
    static getGrantOptions(identity, statement) {
        const actions = typeof (statement.Action) === 'string' ? [statement.Action] : statement.Action;
        const resourceArns = typeof (statement.Resource) === 'string' ? [statement.Resource] : statement.Resource;
        const conditions = statement.Condition ? statement.Condition : {};
        return {
            grantee: identity,
            actions: actions,
            resourceArns: resourceArns,
            conditions: conditions,
        };
    }
    /**
     * Creates role with permissions of activity
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param roleNameSuffix the name suffix of the role that will be created, if empty the role will have the name of the activity.
     * @param roleDescription the description of the role that will be created.
     * @returns - The role that is created with the permissions of the activity
     */
    createRole(scope, id, roleNameSuffix, roleDescription = '') {
        if (!roleNameSuffix || !roleNameSuffix.length) {
            throw TypeError('The role name should be a non empty string');
        }
        const policy = this.createPolicy(scope);
        const role = new iam.Role(scope, id, {
            roleName: `SageMaker-${roleNameSuffix}`,
            description: roleDescription,
            assumedBy: this.createPrincipal(),
        });
        role.attachInlinePolicy(policy);
        return role;
    }
    /**
     * Creates policy with permissions of activity
     * @param scope the Construct scope.
     * @returns - The policy that is created with the permissions of the activity
     */
    createPolicy(scope) {
        const templateFile = util_1.getTemplateFile(this.activityName, this.version, this.isVPCCustomized, this.isKMSCustomized);
        const timestamp = Date.now().toString();
        const templateName = `${templateFile.name}_V${this.version}_${timestamp}`;
        const templateAsString = JSON.stringify(templateFile.templateJson);
        // Replace singleValueReplacements and multiValueReplacements in templateDocument
        const policyDocumentJSON = JSON.parse(util_1.replacePatterns(templateAsString, this.singleValueReplacements, this.multiValueReplacements));
        const policyDocument = iam.PolicyDocument.fromJson(policyDocumentJSON);
        return new iam.Policy(scope, `${templateName} id`, {
            policyName: templateName,
            document: policyDocument,
        });
    }
    /**
     * Creates ML Activity service principal using ML Activity trust template
     * @returns - The service principal of the ML Activity
     */
    createPrincipal() {
        const templateFile = util_1.getTemplateFile(this.activityName, this.version, this.isVPCCustomized, this.isKMSCustomized);
        const templateAsString = JSON.stringify(templateFile.trustTemplateJson);
        // Replace singleValueReplacements and multiValueReplacements in templateDocument
        const policyDocumentJSON = JSON.parse(util_1.replacePatterns(templateAsString, this.singleValueReplacements, this.multiValueReplacements));
        const policyDocumentStatement = policyDocumentJSON.Statement[0];
        return new iam.ServicePrincipal(policyDocumentStatement.Principal.Service, {
            conditions: policyDocumentStatement.Condition,
        });
    }
    /**
     * Grant permissions of activity to identity
     * @param identity identity to be granted permissions
     * @returns - The grant with the permissions granted to the identity
     */
    grantPermissionsTo(identity) {
        const templateFile = util_1.getTemplateFile(this.activityName, this.version, this.isVPCCustomized, this.isKMSCustomized);
        const templateAsString = JSON.stringify(templateFile.templateJson);
        // Replace singleValueReplacements and multiValueReplacements in templateDocument
        let policyDocumentJSON = JSON.parse(util_1.replacePatterns(templateAsString, this.singleValueReplacements, this.multiValueReplacements));
        let grant = iam.Grant.addToPrincipal(Activity.getGrantOptions(identity, policyDocumentJSON.Statement[0]));
        for (let i = 1; i < policyDocumentJSON.Statement.length; i++) {
            grant = grant.combine(iam.Grant.addToPrincipal(Activity.getGrantOptions(identity, policyDocumentJSON.Statement[i])));
        }
        return grant;
    }
    customizeVPC(subnets, securityGroups) {
        if (!this.isCustomizationAvailableForVPC || (!subnets && !securityGroups)) {
            return;
        }
        if (!subnets || !subnets.length) {
            throw TypeError('The array subnets must be of type ec2.ISubnet[] with at least one element.');
        }
        if (!securityGroups || !securityGroups.length) {
            throw TypeError('The array securityGroups must be of type ec2.ISecurityGroup[] with at least one element.');
        }
        this.multiValueReplacements.set(Activity.SUBNETS_PARAMETER_NAME, subnets.map((subnet) => subnet.subnetId));
        this.multiValueReplacements.set(Activity.SECURITY_GROUPS_PARAMETER_NAME, securityGroups.map((securityGroup) => securityGroup.securityGroupId));
        this.isVPCCustomized = true;
    }
    customizeKMS(dataKeys, volumeKeys) {
        if (!this.isCustomizationAvailableForKMS || (!dataKeys && !volumeKeys)) {
            return;
        }
        if (!dataKeys || !dataKeys.length) {
            throw TypeError('The array dataKeys must be of type kms.IKey[] with at least one element.');
        }
        if (!volumeKeys || !volumeKeys.length) {
            throw TypeError('The array volumeKeys must be of type kms.IKey[] with at least one element.');
        }
        this.multiValueReplacements.set(Activity.DATA_KEYS_PARAMETER_NAME, dataKeys.map((key) => key.keyId));
        this.multiValueReplacements.set(Activity.VOLUME_KEYS_PARAMETER_NAME, volumeKeys.map((key) => key.keyId));
        this.isKMSCustomized = true;
    }
}
exports.Activity = Activity;
_a = JSII_RTTI_SYMBOL_1;
Activity[_a] = { fqn: "@cdklabs/cdk-aws-sagemaker-role-manager.Activity", version: "0.0.20" };
// Activity Default Names
Activity.ACCESS_AWS_SERVICES = 'SM_ComputeExecutionRole';
Activity.ACCESS_S3_ALL_RESOURCES = 'SageMakerS3AllResourcesPolicyTemplate';
Activity.ACCESS_S3_BUCKETS = 'SageMakerS3BucketPolicyTemplate';
Activity.MANAGE_ENDPOINTS_ACTIVITY_NAME = 'SM_EndpointDeployment';
Activity.MANAGE_EXPERIMENTS_ACTIVITY_NAME = 'SM_ExperimentsManagement';
Activity.MANAGE_GLUE_TABLES_ACTIVITY_NAME = 'SM_GlueTableManagement';
Activity.MANAGE_JOBS_ACTIVITY_NAME = 'SM_CommonJobManagement';
Activity.MANAGE_MODELS_ACTIVITY_NAME = 'SM_ModelManagement';
Activity.MANAGE_PIPELINES_ACTIVITY_NAME = 'SM_PipelineManagement';
Activity.MONITOR_MODELS_ACTIVITY_NAME = 'SM_ModelMonitoring';
Activity.QUERY_ATHENA_WORKGROUPS = 'SM_AthenaQueryAccess';
Activity.RUN_STUDIO_APPS = 'SM_StudioAppPermissions';
Activity.VISUALIZE_EXPERIMENTS = 'SM_ExperimentsVisualization';
// Activity Default Parameter Values
Activity.ATHENA_WORKGROUP_NAMES_DEFAULT_VALUE = ['primary'];
// Single Value Replacement Parameter Names
Activity.ACCOUNT_ID_PARAMETER_NAME = 'AccountId';
Activity.REGION_PARAMETER_NAME = 'Region';
// Multi Value Replacement Parameter Names
Activity.ATHENA_WORKGROUP_NAMES_PARAMETER_NAME = 'WorkGroupNames';
Activity.ECR_REPOSITORIES_PARAMETER_NAME = 'ECRRepoArns';
Activity.GLUE_DATABASE_NAMES_PARAMETER_NAME = 'GlueDbNames';
Activity.PASSED_ROLES_PARAMETER_NAME = 'PassRoles';
Activity.S3_BUCKETS_PARAMETER_NAME = 'S3Buckets';
Activity.SUBNETS_PARAMETER_NAME = 'VpcSubnets';
Activity.SECURITY_GROUPS_PARAMETER_NAME = 'VpcSecurityGroups';
Activity.DATA_KEYS_PARAMETER_NAME = 'DataKmsKeys';
Activity.VOLUME_KEYS_PARAMETER_NAME = 'VolumeKmsKeys';
//# sourceMappingURL=data:application/json;base64,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