"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTemplateFile = exports.replacePatterns = exports.parameterValidationRegex = exports.TEMPLATES_DIR = void 0;
const fs = require("fs");
const path = require("path");
exports.TEMPLATES_DIR = '../assets/templates';
exports.parameterValidationRegex = /^[A-Za-z0-9\.\/:\-_]+$/;
/**
 * Return the template by replacing its placeholders with the corresponding single value replacements and multi value replacements
 * @param template - The template in which to replace the placeholders
 * @param singleValueReplacements - The map of single value replacements
 * @param multiValueReplacements - The map of multi value replacements
 * @returns - The template file with all placeholders replaced with their corresponding single value and multi value replacements
 */
function replacePatterns(template, singleValueReplacements, multiValueReplacements) {
    // Validate that all single value placeholders of template exist in singleValueReplacements
    const singleValuePlaceholders = template.match(new RegExp('\{\{.*?}}')) || [];
    for (const singleValuePlaceholder of singleValuePlaceholders) {
        if (!singleValueReplacements.has(singleValuePlaceholder.slice(2, singleValuePlaceholder.length - 2))) {
            throw TypeError(`A single value placeholder is missing. Please include a value for ${singleValuePlaceholder}`);
        }
    }
    // Validate that all multi value placeholders of template exist in multiValueReplacements
    const multiValuePlaceholders = template.match(new RegExp('\\[\\[.*?]]')) || [];
    for (const multiValuePlaceholder of multiValuePlaceholders) {
        if (!multiValueReplacements.has(multiValuePlaceholder.slice(2, multiValuePlaceholder.length - 2))) {
            throw TypeError(`A multi value placeholder is missing. Please include a value for ${multiValuePlaceholder}`);
        }
    }
    let ret = `${template}`;
    // Replace all single value placeholders in template
    singleValueReplacements.forEach((value, key) => {
        // Regex matching single value replacement brackets and placeholder
        const singleValueRegularExpression = new RegExp(`\{\{${key}}}`, 'g');
        ret = ret.replace(singleValueRegularExpression, value);
    });
    // Replace all multi value placeholders in template
    multiValueReplacements.forEach((values, key) => {
        // Regex matching a string with multi value replacement brackets and placeholder key in them
        // Example: 'arn:aws:s3:::[[S3Buckets]]/*'
        const multiValueRegularExpression = new RegExp(`\"((?!\").)*\\[\\[${key}]].*?\"`, 'g');
        // All strings matching with multi value replacement brackets and placeholder key in them
        const matches = ret.match(multiValueRegularExpression) || [];
        for (const match of matches) {
            // Replacement string containing a comma separated string of the matched string templated with the multi value replacement values
            const replacementValueString = values.map((value) => match.replace(new RegExp(`\\[\\[${key}]]`, 'g'), value)).join(',');
            // Replace all matches with replacementValueString
            ret = ret.replace(match, replacementValueString);
        }
    });
    return ret;
}
exports.replacePatterns = replacePatterns;
/**
 * Return the template file of the activity with activityName
 * @param activityName - The name of the activity to return the template
 * @param version - The version of the activity to return the template
 * @param isVPCCustomized - Whether VPC is customized
 * @param isKMSCustomized - Whether KMS is customized
 * @returns - The template file
 */
function getTemplateFile(activityName, version, isVPCCustomized = false, isKMSCustomized = false) {
    // Get template name for activity taking into consideration vpc and kms customization
    let templateName = `${activityName}`;
    templateName = isVPCCustomized ? `${templateName}_VPC` : templateName;
    templateName = isKMSCustomized ? `${templateName}_KMS` : templateName;
    const fileNames = fs.readdirSync(path.resolve(__dirname, exports.TEMPLATES_DIR));
    for (const fileName of fileNames) {
        const jsonFileContents = JSON.parse(fs.readFileSync(path.resolve(__dirname, `${exports.TEMPLATES_DIR}/${fileName}`), 'utf8'));
        if (jsonFileContents.name === templateName && jsonFileContents.version == version) {
            return jsonFileContents;
        }
    }
    throw new RangeError(`The activityName did not match any template name field: ${activityName}`);
}
exports.getTemplateFile = getTemplateFile;
//# sourceMappingURL=data:application/json;base64,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