"""Output schema implementation for FEP."""

from dataclasses import dataclass
from typing import Any


@dataclass
class Output:
    """
    Contains the output generated by the system being evaluated.

    The output generation process is treated as a black box by the protocol.
    The output, combined with the test case, forms the complete evaluation context.

    Required Fields:
    - value: The actual output from the system being evaluated

    Optional Fields:
    - id: Unique identifier for this output instance
    - metadata: System-specific information about the output generation
    """

    value: str | dict[str, Any] | None
    id: str | None = None
    metadata: dict[str, Any] | None = None

    def __post_init__(self):
        """Validate required fields and constraints."""
        if self.value is not None and not isinstance(self.value, str | dict):
            raise ValueError("Output.value must be a string, dictionary, or None")

        if self.id is not None and (not isinstance(self.id, str) or not self.id.strip()):
            raise ValueError("Output.id must be a non-empty string if provided")
