#!/usr/bin/env python3
"""
otpylib-pool: Worker Pool Management for OTPylib

A poolboy-inspired worker pool implementation for Python using OTPylib's
dynamic supervisor and gen_server patterns.

Provides:
- Fixed-size and dynamic worker pools
- Overflow workers for handling spikes
- Worker checkout/checkin with timeouts
- Automatic worker lifecycle management
- Integration with otpylib supervision trees
- Transaction-style worker usage (with context manager)

Example:
    ```python
    from otpylib_pool import start, PoolSpec, Pool
    
    # Define pool specification
    pool_spec = PoolSpec(
        name="db_pool",
        worker_module=DatabaseWorker,
        worker_args={"db_url": "postgresql://..."},
        size=10,
        max_overflow=5,
        strategy="fifo"
    )
    
    # Start pool supervisor tree
    pool_sup_pid = await start(pool_spec)
    
    # Checkout worker
    async with Pool.transaction("db_pool_manager") as worker_pid:
        result = await gen_server.call(worker_pid, ("query", "SELECT * FROM users"))
    ```
"""

from .lifecycle import start, stop
from .data import PoolSpec, PoolStatus, WorkerInfo, CheckoutRequest, PoolManagerState
from .client import Pool

# Export commonly used atoms
from .atoms import (
    # RPC atoms
    CHECKOUT,
    CHECKIN, 
    STATUS,
    WHICH_WORKERS,
    PING,
    
    # Response atoms
    OK,
    ERROR,
    TIMEOUT,
    FULL,
    NOPROC,
    
    # Worker state atoms
    AVAILABLE,
    BUSY,
    OVERFLOW,
    
    # Strategy atoms
    FIFO,
    LIFO,
    
    # Lifecycle atoms
    POOL_MANAGER,
    POOL_SUP,
    WORKER_SUP,
)

__version__ = "0.1.0"

__all__ = [
    # Lifecycle
    "start",
    "stop",
    
    # Client API
    "Pool",
    
    # Data structures
    "PoolSpec",
    "PoolStatus",
    "WorkerInfo",
    "CheckoutRequest",
    "PoolManagerState",
    
    # RPC atoms
    "CHECKOUT",
    "CHECKIN",
    "STATUS",
    "WHICH_WORKERS",
    "PING",
    
    # Response atoms
    "OK",
    "ERROR",
    "TIMEOUT",
    "FULL",
    "NOPROC",
    
    # Worker state atoms
    "AVAILABLE",
    "BUSY",
    "OVERFLOW",
    
    # Strategy atoms
    "FIFO",
    "LIFO",
    
    # Lifecycle atoms
    "POOL_MANAGER",
    "POOL_SUP",
    "WORKER_SUP",
]
