#!/usr/bin/env python3
"""
Pool Manager GenServer

The core GenServer that manages worker pool state, checkout/checkin operations,
and worker lifecycle. Uses atom-based message dispatch for high performance.
"""

import time
from typing import Any
from otpylib import atom, process, gen_server
from otpylib.module import OTPModule, GEN_SERVER
from otpylib.gen_server.data import Reply, NoReply, Stop

from result import Result

from otpylib_pool.atoms import (
    CHECKOUT, CHECKIN, STATUS, WHICH_WORKERS, PING, STOP as STOP_ATOM,
    WORKER_EXITED, CHECKOUT_TIMEOUT, DOWN,
    OK, ERROR, TIMEOUT, NOPROC,
)
from otpylib_pool.data import PoolManagerState, PoolSpec, CheckoutRequest
from otpylib_pool import core


# =============================================================================
# Pool Manager GenServer Module
# =============================================================================

class PoolManager(metaclass=OTPModule, behavior=GEN_SERVER, version="1.0.0"):
    """
    Pool Manager GenServer.
    
    Manages a pool of workers with checkout/checkin semantics, overflow
    support, and automatic worker lifecycle management.
    
    Uses atom-based message dispatch for high performance.
    """
    
    async def init(self, pool_spec: PoolSpec):
        """Initialize pool manager with worker supervisor."""
        # Create initial state from spec
        state = PoolManagerState(
            pool_name=pool_spec.name,
            worker_module=pool_spec.worker_module,
            worker_args=pool_spec.worker_args,
            size=pool_spec.size,
            max_overflow=pool_spec.max_overflow,
            strategy=pool_spec.strategy,
        )
        
        # Initialize worker supervisor
        result: Result = await core.initialize_worker_supervisor(state)
        if result.is_err():
            err = result.unwrap_err()
            raise Exception(err)  # Let supervisor treat this as init crash
        
        worker_sup_pid = result.unwrap()
        state.worker_supervisor_pid = worker_sup_pid
        
        # Start initial pool workers
        result = await core.start_initial_workers(state)
        if result.is_err():
            err = result.unwrap_err()
            raise Exception(err)
        
        workers_started = result.unwrap()
        
        return state
    
    async def handle_call(self, message, from_, state: PoolManagerState):
        """Handle synchronous pool operations using atom dispatch."""
        match message:
            # Checkout a worker from the pool
            case (checkout_atom, timeout) if checkout_atom == CHECKOUT:
                result = await core.checkout_worker(from_, timeout, state)
                
                if result.is_ok():
                    worker_pid = result.unwrap()
                    return (Reply(payload=(OK, worker_pid)), state)
                else:
                    error = result.unwrap_err()
                    # If no workers available and queue is full, return error immediately
                    if error == "timeout":
                        return (Reply(payload=(TIMEOUT, None)), state)
                    else:
                        # Queue the request and schedule timeout
                        checkout_req = CheckoutRequest(
                            from_pid=from_,
                            requested_at=time.monotonic(),
                            timeout=timeout
                        )
                        state.waiting_checkouts.append(checkout_req)
                        
                        # Schedule timeout if specified
                        if timeout is not None:
                            my_pid = process.self()
                            await process.send_after(timeout, my_pid, (CHECKOUT_TIMEOUT, from_))
                        
                        # Return NoReply - we'll reply when worker becomes available
                        return (NoReply(), state)
            
            # Checkin a worker back to the pool
            case (checkin_atom, worker_pid) if checkin_atom == CHECKIN:
                result = await core.checkin_worker(worker_pid, state)
                
                if result.is_ok():
                    return (Reply(payload=OK), state)
                else:
                    error = result.unwrap_err()
                    return (Reply(payload=(ERROR, error)), state)
            
            # Get pool status
            case msg if msg == STATUS:
                result = await core.get_pool_status(state)
                
                if result.is_ok():
                    status = result.unwrap()
                    return (Reply(payload=status), state)
                else:
                    error = result.unwrap_err()
                    return (Reply(payload=(ERROR, error)), state)
            
            # List all workers
            case msg if msg == WHICH_WORKERS:
                result = await core.list_workers(state)
                
                if result.is_ok():
                    workers = result.unwrap()
                    return (Reply(payload=workers), state)
                else:
                    error = result.unwrap_err()
                    return (Reply(payload=(ERROR, error)), state)
            
            # Ping for health check
            case msg if msg == PING:
                return (Reply(payload="pong"), state)
            
            case _:
                return (Reply(payload=(ERROR, f"Unknown call: {message}")), state)
    
    async def handle_cast(self, message, state: PoolManagerState):
        """Handle asynchronous pool operations."""
        match message:
            case msg if msg == STOP_ATOM:
                return (Stop(reason=None), state)
            
            case _:
                return (NoReply(), state)
    
    async def handle_info(self, message, state: PoolManagerState):
        """Handle info messages (worker exits, timeouts, monitors)."""
        match message:
            # Worker process exited
            case (DOWN, ref, pid, reason):
                result = await core.handle_worker_exit(pid, reason, state)
                return (NoReply(), state)
            
            # Checkout timeout expired
            case (CHECKOUT_TIMEOUT, from_pid):
                result = await core.handle_checkout_timeout(from_pid, state)
                return (NoReply(), state)
            
            case _:
                return (NoReply(), state)
    
    async def terminate(self, reason, state: PoolManagerState):
        """Cleanup on termination."""
        # Terminate all workers
        await core.terminate_all_workers(state)
